package com.xebialabs.deployit.booter.local;

import java.lang.annotation.Annotation;
import java.lang.reflect.Method;
import java.util.*;
import java.util.function.Function;
import java.util.stream.Collectors;
import java.util.stream.Stream;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;

import com.xebialabs.deployit.booter.local.validation.CollectionTypeValidator;
import com.xebialabs.deployit.booter.local.validation.ReferenceCollectionTypeValidator;
import com.xebialabs.deployit.booter.local.validation.ReferenceTypeValidator;
import com.xebialabs.deployit.booter.local.validation.RequiredValidator;
import com.xebialabs.deployit.plugin.api.Deprecations;
import com.xebialabs.deployit.plugin.api.reflect.*;
import com.xebialabs.deployit.plugin.api.udm.CandidateValuesFilter;
import com.xebialabs.deployit.plugin.api.udm.ConfigurationItem;
import com.xebialabs.deployit.plugin.api.udm.IDictionary;
import com.xebialabs.deployit.plugin.api.udm.Property;
import com.xebialabs.deployit.plugin.api.validation.ExtendedValidationContext;
import com.xebialabs.deployit.plugin.api.validation.ValidationContext;
import com.xebialabs.deployit.plugin.api.validation.ValidationMessage;
import com.xebialabs.deployit.plugin.api.validation.Validator;

import nl.javadude.scannit.Scannit;

import static com.xebialabs.deployit.booter.local.utils.CheckUtils.checkArgument;
import static com.xebialabs.deployit.plugin.api.reflect.PropertyKind.CI;
import static com.xebialabs.deployit.plugin.api.reflect.PropertyKind.LIST_OF_CI;
import static com.xebialabs.deployit.plugin.api.reflect.PropertyKind.LIST_OF_STRING;
import static com.xebialabs.deployit.plugin.api.reflect.PropertyKind.MAP_STRING_STRING;
import static com.xebialabs.deployit.plugin.api.reflect.PropertyKind.SET_OF_CI;
import static com.xebialabs.deployit.plugin.api.reflect.PropertyKind.SET_OF_STRING;
import static com.xebialabs.deployit.plugin.api.reflect.PropertyKind.STRING;
import static java.lang.String.format;
import static java.util.EnumSet.of;

public abstract class LocalPropertyDescriptor implements PropertyDescriptor {
    private static final Integer ZERO = 0;
    protected final Logger logger = LoggerFactory.getLogger(getClass());

    private final DescriptorRegistryId typeSource;

    private LocalDescriptor declaringDescriptor;
    private String fullyQualifiedName;
    private String name;

    private boolean asContainment;
    private boolean nested;
    private int order;
    private String category;
    private String description;
    private String label;
    private boolean password;
    private boolean required;
    private Property.Size size;
    private PropertyKind kind;
    private List<String> enumValues;

    private Class<?> enumClass;
    private Type referencedType;
    private boolean hidden;
    private boolean inspectionProperty;
    private boolean requiredForInspection;
    private boolean isTransient;
    private boolean readonly;
    protected Set<Validator<?>> validationRules = new HashSet<>();
    private Set<String> aliases = new HashSet<>();
    private String candidateValuesFilter;
    private boolean deployedSpecific;
    private boolean primitive;
    private InputHint inputHint;
    private List<Annotation> annotations = new ArrayList<>();

    protected LocalPropertyDescriptor(DescriptorRegistryId typeSource) {
        this.typeSource = typeSource;
    }

    protected void reInitializeRequired() {
        if (kind == PropertyKind.BOOLEAN && required) {
            logger.debug("Required boolean property [{}] will be treated as an optional property.", this);
            required = false;
        }

        if ((kind == PropertyKind.LIST_OF_CI || kind == PropertyKind.SET_OF_CI) && asContainment && required) {
            logger.debug("Required asContainment collection of configuration items [{}] will be treated as an optional property.", this);
            required = false;
        }
    }

    LocalPropertyDescriptor copyWithNewDescriptor(LocalDescriptor declaringDescriptor) {
        return new LocalPropertyDescriptorWithDifferentOwner(this, declaringDescriptor);
    }

    protected void setFromPropertyDescriptor(LocalPropertyDescriptor pd) {
        this.setName(pd.getName());
        this.setAsContainment(pd.isAsContainment());
        this.setNested(pd.isNested());
        this.setOrder(pd.getOrder());
        this.setCategory(pd.getCategory());
        this.setDescription(pd.getDescription());
        this.setLabel(pd.getLabel());
        this.setPassword(pd.isPassword());
        this.setRequired(pd.isRequired());
        this.setSize(pd.getSize());
        this.setKind(pd.getKind());
        this.setEnumValues(pd.getEnumValues());
        this.setEnumClass(pd.getEnumClass());
        this.setReferencedType(pd.getReferencedType());
        this.setHidden(pd.isHidden());
        this.setInspectionProperty(pd.isInspectionProperty());
        this.setRequiredForInspection(pd.isRequiredForInspection());
        this.setValidationRules(new HashSet<>(pd.validationRules));
        this.setAliases(new HashSet<>(pd.getAliases()));
        this.setTransient(pd.isTransient());
        this.setCandidateValuesFilter(pd.getCandidateValuesFilter());
        this.setDeployedSpecific(pd.isDeployedSpecific());
        this.setInputHint(pd.getInputHint());
        this.setReadonly(pd.isReadonly());
    }

    public void verify(Verifications verifications) {
        verifyName(verifications, "name");
        // UI naming convention.
        verifyName(verifications, "displayName");
        verifyName(verifications, "id");
        verifyName(verifications, "type");
        verifications.verify(!getName().contains("$"), "Cannot define a property named '%s' because it contains a '$'", getName());
        verifications.verify(!hidden || isTransient, "Hidden property '%s' should be transient", this);

        boolean isCiKind = of(LIST_OF_CI, SET_OF_CI, CI).contains(kind);
        verifications.verify(!asContainment || isCiKind, "'%s' can only be an as containment relation if it is a 'set_of_ci', 'list_of_ci' or 'ci' kind property", this);
        verifications.verify(!nested || kind == CI, "'%s' can only be nested if it is a 'ci' kind property", this);
        verifications.verify(referencedType == null || isCiKind, "'%s' can only have reference type '%s' set when it is a (Set/List of) CI kind property", this, referencedType);
        verifications.verify(candidateValuesFilter == null || isCiKind, "'%s' can only define a candidate-values-filter if it is a 'set_of_ci', 'list_of_ci' or 'ci' kind property", this);
        verifications.verify(candidateValuesFilter == null || findFilterMethod() != null, "'%s' refers to an unknown candidate-values-filter '%s'", this, candidateValuesFilter);
        verifications.verify(referencedType == null || isCiKind, "'%s' can only have reference type '%s' set when it is a (Set/List of) CI kind property", this, referencedType);
        boolean isPasswordKind = of(STRING, SET_OF_STRING, LIST_OF_STRING, MAP_STRING_STRING).contains(kind);
        if (password && !isPasswordKind) {
            Deprecations.deprecated("'%s' can only be a password property if it is a 'string', 'set_of_string', 'list_of_string' or 'map_string_string' kind property", this);
        }

        verifyAliases(verifications);
    }

    private void verifyAliases(Verifications verifications) {
        if (!aliases.isEmpty()) {
            for (PropertyDescriptor otherPd : declaringDescriptor.getPropertyDescriptors()) {
                LocalPropertyDescriptor pd = (LocalPropertyDescriptor) otherPd;
                if (!otherPd.equals(this)) {
                    verifications.verify(declaringDescriptor.getType(), !aliases.contains(pd.getName()), "Aliases of [%s] contain name [%s] which is an existing property.", this, pd.getName());
                    Set<String> intersection = new HashSet<>(aliases);
                    intersection.retainAll(pd.aliases);
                    verifications.verify(declaringDescriptor.getType(), intersection.isEmpty(), "Aliases of [%s] conflict with aliases of [%s]. Conflicting: %s", this, pd, intersection);
                }
            }
        }
    }

    private void verifyName(Verifications verifications, String name) {
        verifications.verify(!this.name.equals(name), "Cannot define a property named '%s' on %s", name, declaringDescriptor);
    }


    private Method findFilterMethod() {
        Set<Method> filters = Scannit.getInstance().getMethodsAnnotatedWith(CandidateValuesFilter.class);
        return filters.stream().filter(i -> i.getAnnotation(CandidateValuesFilter.class).name().equals(candidateValuesFilter)).findFirst().orElse(null);
    }

    protected void initEnumValues(Class<?> enumClass) {
        initEnumValues(enumClass, Collections.emptyList());
    }

    void initEnumValues(Class<?> enumClass, List<String> allowedValues) {
        enumValues = new ArrayList<>();
        this.enumClass = enumClass;
        for (Enum<?> enumValue : (Enum<?>[]) enumClass.getEnumConstants()) {
            if (allowedValues.isEmpty() || allowedValues.contains(enumValue.name())) {
                enumValues.add(enumValue.name());
            }
        }
    }

    protected void addDefaultValidationRules() {
        if (required && !asContainment && kind != PropertyKind.BOOLEAN) {
            validationRules.add(new RequiredValidator());
        }

        if (of(SET_OF_CI, SET_OF_STRING, LIST_OF_CI, LIST_OF_STRING).contains(kind)) {
            validationRules.add(new CollectionTypeValidator(this));
        }

        if (of(LIST_OF_CI).contains(kind) && this.getReferencedType().getName().equals("IDictionary")) {
            validationRules.add((value, context) -> {
                final LinkedHashSet dictionariesSet = new LinkedHashSet();

                String duplicateDictionaries = ((Collection<?>) value).stream().
                        filter(x -> !dictionariesSet.add(x)).
                        map(n -> ((IDictionary) n).getName()).
                        collect(Collectors.joining(", "));

                if (!duplicateDictionaries.isEmpty()) {
                    context.error("Property [%s] has duplicate values: [%s]. Please remove the" +
                                    " duplicate entries to proceed.", this.getName(),
                            duplicateDictionaries);
                }

            });
        }

        if (of(SET_OF_CI, LIST_OF_CI).contains(kind)) {
            validationRules.add(new ReferenceCollectionTypeValidator(this));
        } else if (CI == kind) {
            validationRules.add(new ReferenceTypeValidator(this));
        }
    }

    
    @Override
    public String getName() {
        return name;
    }

    @Override
    public String getDescription() {
        return description;
    }

    @Override
    public boolean isAsContainment() {
        return asContainment;
    }

    @Override
    public boolean isNested() {
        return nested;
    }

    @Override
    public int getOrder() {
        return order;
    }

    @Override
    public String getCategory() {
        return category;
    }

    @Override
    public String getLabel() {
        return label;
    }

    @Override
    public boolean isPassword() {
        return password;
    }

    @Override
    public boolean isRequired() {
        return required;
    }

    @Override
    public Property.Size getSize() {
        return size;
    }

    @Override
    public PropertyKind getKind() {
        return kind;
    }

    @Override
    public List<String> getEnumValues() {
        return enumValues != null ? new ArrayList<>(enumValues) : enumValues;
    }

    public Class<?> getEnumClass() {
        return enumClass;
    }

    public LocalDescriptor getDeclaringDescriptor() {
        return declaringDescriptor;
    }

    @Override
    public Type getReferencedType() {
        return referencedType;
    }

    @Override
    public Object getDefaultValue() {
        var globalValue = GlobalContextRegistry.lookup(this);
        return convertValue(globalValue);
    }

    protected void registerDefault(String defaultValue) {
        // at this point type is associated with descriptor registry
        // and we can ask GlobalContextRegistry for a global context
        GlobalContextRegistry.register(this, defaultValue);
    }

    protected void registerDefault(PropertyDescriptor superPropertyDescriptor) {
        GlobalContextRegistry.register(this, superPropertyDescriptor);
    }

    Converter createConverter() {
        if (kind == PropertyKind.ENUM) {
            if (enumClass != null) {
                return new EnumClassConverter(enumClass);
            } else {
                return new EnumValuesConverter(enumValues);
            }
        } else {
            return Converters.createStatelessConverter(name, kind);
        }
    }

    private Object convertValue(final String val) {
        if (val == null) return null;
        return createConverter().convert(val);
    }

    private Collection<String> splitValue(String val) {
        return Arrays.stream(val.split(",")).map(String::trim).filter(s -> s.length() > 0).collect(Collectors.toList());
    }

    @Override
    public Set<String> getAliases() {
        return aliases;
    }

    @Override
    public String getCandidateValuesFilter() {
        return candidateValuesFilter;
    }

    @Override
    public boolean isHidden() {
        return hidden;
    }

    @Override
    public void set(ConfigurationItem item, Object value) {
        checkIfReadonlyUpdated(this.get(item), value);
        if (value instanceof String) {
            value = convertValue((String) value);
        } else if (value == null) {
            value = getDefaultValue();
        }
        if (value == null) {
            value = emptyValue();
        }
        logger.trace("Setting value [{}] on property [{}] of CI [{}]", value, getFqn(), item.getId());
        doSetValue(item, value);
    }

    private void checkIfReadonlyUpdated(Object currentValue, Object newValue) {
        if (this.isReadonly()) {
            if (!areValuesEqual(currentValue, newValue)) {
                boolean currentValueIsEmpty = currentValue == null || areValuesEqual(currentValue, emptyValue());
                if (!currentValueIsEmpty) {
                    throw new IllegalArgumentException(String.format("Readonly property [%s] cannot be updated", this.getName()));
                }
            }
        }
    }

    private boolean areValuesEqual(Object itemValue,
                                   Object otherValue) {
        return areValuesEqual(itemValue, otherValue, ConfigurationItem::getId, new HashSet<>());
    }

    protected abstract void doSetValue(ConfigurationItem item, Object value);

    @Override
    public boolean areEqual(ConfigurationItem item, ConfigurationItem other) {
        return areEqual(item, other, ConfigurationItem::getId);
    }

    @Override
    public boolean areEqual(ConfigurationItem item,
                            ConfigurationItem other,
                            Function<ConfigurationItem, Object> identifierExtractor) {
        return areEqual(item, other, identifierExtractor, new HashSet<>());
    }

    boolean areEqual(ConfigurationItem item,
                     ConfigurationItem other,
                     Function<ConfigurationItem, Object> identifierExtractor,
                     Set<String> itemsBeingCompared) {
        Object left = get(item);
        Object right = get(other);
        logger.trace("Comparing {}: old [{}] <-> new [{}]", getFqn(), left, right);
        return areValuesEqual(left, right, identifierExtractor, itemsBeingCompared);
    }

    private boolean areEqualByIdentifier(ConfigurationItem left,
                                         ConfigurationItem right,
                                         Function<ConfigurationItem, Object> identifierExtractor) {
        return identifierExtractor.apply(left).equals(identifierExtractor.apply(right));
    }

    @SuppressWarnings("unchecked")
    private boolean areValuesEqual(Object itemValue,
                                   Object otherValue,
                                   Function<ConfigurationItem, Object> identifierExtractor,
                                   Set<String> itemsBeingCompared) {
        if (itemValue == null) {
            return otherValue == null;
        } else if (otherValue == null) {
            return false;
        }

        switch (kind) {
            case SET_OF_STRING:
                Set<String> symmetricDiff = symmetricDifference((Set<String>) itemValue, (Set<String>) otherValue);
                return symmetricDiff.isEmpty();
            case SET_OF_CI:
                Stream<ConfigurationItem> cis = Stream.concat(((Set<ConfigurationItem>) itemValue).stream(), ((Set<ConfigurationItem>) otherValue).stream());
                Map<Object, List<ConfigurationItem>> index = cis.collect(Collectors.groupingBy(identifierExtractor, Collectors.toList()));
                for (Object key : index.keySet()) {
                    Collection<ConfigurationItem> cisToCompare = index.get(key);
                    if (cisToCompare.size() != 2) {
                        return false;
                    }
                    Iterator<ConfigurationItem> itemIterator = cisToCompare.iterator();
                    ConfigurationItem lhs = itemIterator.next();
                    ConfigurationItem rhs = itemIterator.next();
                    if (areCiEqual(lhs, rhs, identifierExtractor, itemsBeingCompared)) continue;
                    return false;
                }
                return true;
            case CI:
                ConfigurationItem itemValueAsCi = (ConfigurationItem) itemValue;
                ConfigurationItem otherValueAsCi = (ConfigurationItem) otherValue;
                if (areEqualByIdentifier(itemValueAsCi, otherValueAsCi, identifierExtractor)) {
                    LocalDescriptor descriptor = (LocalDescriptor) DescriptorRegistry.getDescriptor(itemValueAsCi.getType());
                    return LocalDescriptorHelper.areEqualDeeply(descriptor, itemValueAsCi, otherValueAsCi, itemsBeingCompared);
                }
                return false;
            case MAP_STRING_STRING:
                Map<String, String> left = (Map<String, String>) itemValue;
                Map<String, String> right = (Map<String, String>) otherValue;
                Set<String> diff = symmetricDifference(left.keySet(), right.keySet());
                if (!diff.isEmpty()) {
                    return false;
                }
                for (Map.Entry<String, String> leftKV : left.entrySet()) {
                    if (!leftKV.getValue().equals(right.get(leftKV.getKey()))) {
                        return false;
                    }
                }
                return true;
            case LIST_OF_STRING:
                List<String> leftStrings = (List<String>) itemValue;
                List<String> rightStrings = (List<String>) otherValue;
                // Don't compare the lists using equals, because we might be dealing with different kinds of lists.
                if (leftStrings.size() != rightStrings.size()) {
                    return false;
                }

                for (int i = 0; i < leftStrings.size(); i++) {
                    if (!leftStrings.get(i).equals(rightStrings.get(i))) {
                        return false;
                    }
                }
                return true;
            case LIST_OF_CI:
                List<ConfigurationItem> lhs = (List<ConfigurationItem>) itemValue;
                List<ConfigurationItem> rhs = (List<ConfigurationItem>) otherValue;
                if (lhs.size() != rhs.size()) {
                    return false;
                } else {
                    Iterator<ConfigurationItem> lIter = lhs.iterator();
                    Iterator<ConfigurationItem> rIter = rhs.iterator();
                    ConfigurationItem lItem;
                    ConfigurationItem rItem;
                    while (lIter.hasNext()) {
                        lItem = lIter.next();
                        rItem = rIter.next();
                        if (areCiEqual(lItem, rItem, identifierExtractor, itemsBeingCompared)) continue;
                        return false;
                    }
                    return true;
                }
            default:
                return itemValue.equals(otherValue);
        }
    }

    private Set<String> symmetricDifference(Set<String> leftSet, Set<String> rightSet) {
        Set<String> symmetricDiff = new HashSet<>(leftSet);
        symmetricDiff.addAll(rightSet);
        Set<String> tmp = new HashSet<>(leftSet);
        tmp.retainAll(rightSet);
        symmetricDiff.removeAll(tmp);
        return symmetricDiff;
    }

    private boolean areCiEqual(ConfigurationItem lItem,
                               ConfigurationItem rItem,
                               Function<ConfigurationItem, Object> identifierExtractor,
                               Set<String> itemsBeingCompared) {
        if (areEqualByIdentifier(lItem, rItem, identifierExtractor) && lItem.getType().equals(rItem.getType())) {
            LocalDescriptor descriptor = (LocalDescriptor) DescriptorRegistry.getDescriptor(lItem.getType());
            return LocalDescriptorHelper.areEqualDeeply(descriptor, lItem, rItem, itemsBeingCompared);
        }
        return false;
    }

    @SuppressWarnings("unchecked")
    void validate(final ConfigurationItem ci, final List<ValidationMessage> messages) {
        ValidationContext context = (message, params) -> messages.add(new ValidationMessage(ci.getId(), name, format(message, params)));
        for (Validator<?> validationRule : validationRules) {
            ((Validator<Object>) validationRule).validate(get(ci), context);
        }
    }

    @SuppressWarnings("unchecked")
    void validate(final ExtendedValidationContext context, final ConfigurationItem ci) {
        for (Validator<?> validationRule : validationRules) {
            ((Validator<Object>) validationRule).validate(get(ci), context.focus(ci, name));
        }
    }

    @SuppressWarnings("unchecked")
    void validateInputHint(final ConfigurationItem configurationItem, final List<ValidationMessage> validationMessages) {
        if (inputHint != null) {
            ValidationContext warningContext = (validationMessage, params) -> validationMessages.add(ValidationMessage.warn(configurationItem.getId(), name, format(validationMessage, params)));
            for (final Validator validationRule : inputHint.getValidationRules()) {
                validationRule.validate(get(configurationItem), warningContext);
            }
        }
    }

    @Override
    public boolean isInspectionProperty() {
        return inspectionProperty;
    }

    @Override
    public boolean isRequiredForInspection() {
        return requiredForInspection;
    }

    @Override
    public boolean isTransient() {
        return isTransient;
    }

    @Override
    public boolean isReadonly() {
        return readonly;
    }

    @Override
    public String getFqn() {
        if (fullyQualifiedName == null) {
            assignFullyQualifiedName();
        }
        return fullyQualifiedName;
    }

    private void assignFullyQualifiedName() {
        this.fullyQualifiedName = declaringDescriptor.getType() + "." + name;
    }

    public boolean isDeployedSpecific() {
        return deployedSpecific;
    }

    @Override
    public InputHint getInputHint() {
        return inputHint;
    }

    @Override
    public List<Annotation> getAnnotations() {
        return annotations;
    }

    @Override
    public String toString() {
        return getFqn();
    }

    public Object emptyValue() {
        if (primitive) {
            switch (kind) {
                case INTEGER:
                    return ZERO;
                case BOOLEAN:
                    return Boolean.FALSE;
                default:
                    throw new IllegalArgumentException("Only INTEGER and BOOLEAN can be primitive");
            }
        }
        switch (kind) {
            case SET_OF_STRING:
            case SET_OF_CI:
                return new HashSet<>();
            case LIST_OF_STRING:
            case LIST_OF_CI:
                return new ArrayList<>();
            case MAP_STRING_STRING:
                return new HashMap<>();
            default:
                return null;
        }
    }

    protected void setName(String name) {
        this.name = name;
    }

    protected void setAsContainment(boolean asContainment) {
        this.asContainment = asContainment;
    }

    protected void setNested(boolean nested) {
        this.nested = nested;
    }

    protected void setOrder(int order) {
        this.order = order;
    }


    protected void setCategory(String category) {
        this.category = category;
    }

    protected void setDescription(String description) {
        this.description = description;
    }

    protected void setLabel(String label) {
        this.label = label;
    }

    protected void setPassword(boolean password) {
        this.password = password;
    }

    protected void setRequired(boolean required) {
        this.required = required;
    }

    protected void setSize(Property.Size size) {
        this.size = size;
    }

    protected void setKind(PropertyKind kind) {
        this.kind = kind;
        this.primitive = false;
    }

    protected void setPrimitiveKind(PropertyKind kind) {
        checkArgument(kind == PropertyKind.INTEGER || kind == PropertyKind.BOOLEAN, "Only INTEGER and BOOLEAN can be primitive kinds");
        this.kind = kind;
        this.primitive = true;
    }

    protected void setEnumValues(List<String> enumValues) {
        this.enumValues = enumValues;
    }

    protected void setEnumClass(Class<?> enumClass) {
        this.enumClass = enumClass;
    }

    protected void setHidden(boolean hidden) {
        this.hidden = hidden;
    }

    protected void setReferencedType(Type referencedType) {
        this.referencedType = referencedType;
    }

    protected void setInspectionProperty(boolean inspectionProperty) {
        this.inspectionProperty = inspectionProperty;
    }

    protected void setTransient(boolean aTransient) {
        isTransient = aTransient;
    }

    protected void setReadonly(boolean aReadonly) {
        readonly = aReadonly;
    }

    protected void setCandidateValuesFilter(String candidateValuesFilter) {
        this.candidateValuesFilter = candidateValuesFilter;
    }

    protected void setDeployedSpecific(boolean deployedSpecific) {
        this.deployedSpecific = deployedSpecific;
    }

    protected void setDeclaringDescriptor(LocalDescriptor declaringDescriptor) {
        this.declaringDescriptor = declaringDescriptor;
    }

    protected void setRequiredForInspection(boolean requiredForInspection) {
        this.requiredForInspection = requiredForInspection;
    }

    protected void setAliases(Set<String> aliases) {
        this.aliases = aliases;
    }

    protected void setValidationRules(Set<Validator<?>> validationRules) {
        this.validationRules = validationRules;
    }

    protected void setInputHint(InputHint inputHint) {
        this.inputHint = inputHint;
    }

    protected void setAnnotations(List<Annotation> annotations) {
        this.annotations = annotations;
    }

    @Override
    public DescriptorRegistryId getTypeSource() {
        return typeSource;
    }
}
