package com.xebialabs.deployit.booter.local;

import com.google.common.collect.HashMultimap;
import com.google.common.collect.Maps;
import com.google.common.collect.Multimap;

import com.xebialabs.deployit.plugin.api.reflect.Descriptor;
import com.xebialabs.deployit.plugin.api.reflect.DescriptorRegistry;
import com.xebialabs.deployit.plugin.api.reflect.Type;
import com.xebialabs.deployit.plugin.api.udm.ConfigurationItem;

import org.slf4j.Logger;
import org.slf4j.LoggerFactory;
import org.w3c.dom.Element;

import java.util.Collection;
import java.util.Collections;
import java.util.Map;

import static com.google.common.base.Preconditions.checkArgument;
import static com.google.common.base.Preconditions.checkState;
import static com.xebialabs.deployit.booter.local.LocalDescriptor.from;

public class LocalDescriptorRegistry extends DescriptorRegistry {
    private final Map<Type, Descriptor> descriptors = Maps.newHashMap();

    private final Multimap<Type, Type> subtypes = HashMultimap.create();

    // N.B.: Logger must be defined at top of file so that other static initializers can access it
    private static Logger logger = LoggerFactory.getLogger(LocalDescriptorRegistry.class);

    static void register(Class<? extends ConfigurationItem> ci) {
        Descriptor descriptor = from(ci);
        logger.debug("Found ConfigurationItem: {}", descriptor);
        register(descriptor);
    }

    static void register(Descriptor descriptor) {
        LocalDescriptorRegistry registry = (LocalDescriptorRegistry) getInstance();
        checkState(!registry.descriptors.containsKey(descriptor.getType()), "The type [%s] is already registered in Deployit.", descriptor.getType());
        registry.descriptors.put(descriptor.getType(), descriptor);
    }

    static void registerSyntheticTypeDefinition(Element syntheticElement) {
        Descriptor descriptor = LocalDescriptor.from(syntheticElement);
        logger.debug("Found ConfigurationItem: {}", descriptor);
        register(descriptor);
    }

    static void registerSyntheticTypeModification(Element typeModificationElement) {
        String typeName = SyntheticHelper.getRequiredStringAttribute(typeModificationElement, "type");
        LocalDescriptor d = (LocalDescriptor) getDescriptor(typeName);
        d.parseTypeModification(typeModificationElement);
    }

    static void registerSubtype(Type supertype, Type subtype) {
        checkState(!supertype.equals(subtype), "Cannot register [%s] as its own subtype.", supertype);
        ((LocalDescriptorRegistry) getInstance()).subtypes.put(supertype, subtype);
    }

    @Override
    protected Collection<Descriptor> _getDescriptors() {
        return Collections.unmodifiableCollection(descriptors.values());
    }

    @Override
    protected Collection<Type> _getSubtypes(Type supertype) {
        return Collections.unmodifiableCollection(subtypes.get(supertype));
    }

    @Override
    protected Descriptor _getDescriptor(Type type) {
        checkState(!descriptors.isEmpty(), "DescriptorRegistry not booted");
        checkArgument(exists(type), "Unknown type [%s]", type);
        return descriptors.get(type);
    }

    @Override
    protected boolean _exists(Type type) {
        return descriptors.containsKey(type);
    }
}
