package com.xebialabs.deployit.booter.local;

import com.google.common.collect.Maps;

import com.xebialabs.deployit.plugin.api.reflect.Type;
import com.xebialabs.deployit.plugin.api.validation.Rule;
import com.xebialabs.deployit.plugin.api.validation.Validator;

import nl.javadude.scannit.Scannit;

import org.w3c.dom.Attr;
import org.w3c.dom.Element;
import org.w3c.dom.NamedNodeMap;

import java.lang.annotation.Annotation;
import java.lang.reflect.InvocationTargetException;
import java.lang.reflect.Method;
import java.util.Map;
import java.util.Set;

import static com.xebialabs.deployit.booter.local.utils.ReflectionUtils.handleInvocationTargetException;
import static com.xebialabs.deployit.booter.local.utils.ReflectionUtils.setField;

class ValidationRuleConverter {

    public static final String RULE_ELEMENT_NAME = "rule";

    private static final Map<String, Class<? extends Validator<?>>> ruleTypeMap = Maps.newHashMap();

    static {
        Set<Class<?>> rules = Scannit.getInstance().getTypesAnnotatedWith(Rule.class);
        for (Class<?> validationRuleAnnotation : rules) {
            Rule rule = validationRuleAnnotation.getAnnotation(Rule.class);
            ruleTypeMap.put(rule.type(), rule.clazz());
        }
    }

    static boolean isRule(Annotation annotation) {
        return annotation.annotationType().isAnnotationPresent(Rule.class);
    }

    static Validator<?> makeRule(Annotation validationRuleAnnotation) {
        Class<? extends Annotation> vraClazz = validationRuleAnnotation.annotationType();
        Rule rule = vraClazz.getAnnotation(Rule.class);
        Validator<?> validationRule = instantiate(rule.clazz());
        for (Method method : vraClazz.getDeclaredMethods()) {
            Object annotationValue = getAnnotationValue(validationRuleAnnotation, method);
            setField(validationRule, method.getName(), annotationValue);
        }
        return validationRule;
    }

    static Validator<?> makeRule(Element element, String fqn) {
        String type = element.getAttribute("type");
        Validator<?> rule = createRule(element, type);
        if (rule != null) {
            return rule;
        }
        throw new IllegalArgumentException(String.format("Could not find validation rule type %s defined on property %s", type, fqn));
    }

    static Validator<?> makeRule(Element element, Type ciType) {
        String type = element.getAttribute("type");
        Validator<?> rule = createRule(element, type);
        if (rule != null) {
            return rule;
        }
        throw new IllegalArgumentException(String.format("Could not find validation rule type %s defined on type %s", type, ciType));
    }

    private static Validator<?> createRule(Element element, String type) {
        if (ruleTypeMap.containsKey(type)) {
            Validator<?> validationRule = instantiate(ruleTypeMap.get(type));
            NamedNodeMap attributes = element.getAttributes();
            for (int i = 0; i < attributes.getLength(); i++) {
                Attr attr = (Attr) attributes.item(i);
                String name = attr.getName();
                if (!name.equals("type")) {
                    setField(validationRule, name, attr.getValue());
                }
            }
            return validationRule;
        }
        return null;
    }

    private static Validator<?> instantiate(Class<? extends Validator<?>> rule) {
        try {
            return rule.newInstance();
        } catch (InstantiationException e) {
            throw new IllegalStateException("Cannot instantiate validation rule " + rule, e);
        } catch (IllegalAccessException e) {
            throw new IllegalStateException("Cannot instantiate validation rule " + rule, e);
        }
    }

    private static Object getAnnotationValue(Annotation validationRuleAnnotation, Method method) {
        try {
            return method.invoke(validationRuleAnnotation);
        } catch (IllegalAccessException e) {
            throw new IllegalStateException("Could not invoke method.", e);
        } catch (InvocationTargetException e) {
            throw handleInvocationTargetException(e, "Could not convert validation rule.");
        }
    }
}
