package com.xebialabs.deployit.booter.local;

import com.xebialabs.deployit.booter.local.utils.ReflectionUtils;
import com.xebialabs.deployit.plugin.api.reflect.Descriptor;
import com.xebialabs.deployit.plugin.api.reflect.MethodDescriptor;
import com.xebialabs.deployit.plugin.api.udm.ConfigurationItem;
import com.xebialabs.deployit.plugin.api.udm.ControlTask;

import org.w3c.dom.Attr;
import org.w3c.dom.Element;
import org.w3c.dom.NamedNodeMap;

import java.lang.reflect.InvocationTargetException;
import java.lang.reflect.Method;
import java.util.List;
import java.util.Map;
import java.util.Set;

import static com.google.common.base.Preconditions.checkState;
import static com.google.common.collect.Maps.newHashMap;
import static com.google.common.collect.Sets.newEnumSet;
import static com.google.common.collect.Sets.newHashSet;

class LocalMethodDescriptor implements MethodDescriptor {

    private static final String CONTROL_TASK_DISPATCH_METHOD = "controlTaskDispatch";
    private static final Set<String> RESERVED_ATTRIBUTE_NAMES = newHashSet("name", "label", "description");

    private String name;
    private String label;
    private Method method;
    private String description;
    private String delegate;
    private Map<String, String> attributes = newHashMap();
    private Descriptor descriptor;

    private LocalMethodDescriptor(Descriptor descriptor, String name) {
        this.descriptor = descriptor;
        this.name = name;
    }

    private LocalMethodDescriptor(Descriptor descriptor, Method method) {
        this.method = method;
        this.name = method.getName();
        this.descriptor = descriptor;
    }

    LocalMethodDescriptor(LocalMethodDescriptor copyOf, Descriptor newOwner) {
        this.method = copyOf.method;
        this.name = copyOf.name;
        this.description = copyOf.description;
        this.label = copyOf.label;
        this.attributes = copyOf.attributes;
        this.descriptor = newOwner;
    }

    static MethodDescriptor from(Descriptor descriptor, Method method) {
        LocalMethodDescriptor methodDescriptor = new LocalMethodDescriptor(descriptor, method);
        methodDescriptor.initMetadata();
        return methodDescriptor;
    }

    static MethodDescriptor from(Descriptor descriptor, Element element) {
        String name = SyntheticHelper.getRequiredStringAttribute(element, "name");
        LocalMethodDescriptor methodDescriptor = new LocalMethodDescriptor(descriptor, name);
        methodDescriptor.label = SyntheticHelper.getOptionalStringAttribute(element, "label", name);
        methodDescriptor.description = SyntheticHelper.getOptionalStringAttribute(element, "description", "No description.");
        methodDescriptor.delegate = SyntheticHelper.getOptionalStringAttribute(element, "delegate", null);
        methodDescriptor.attributes = readAttributes(element);
        return methodDescriptor;
    }

    private static Map<String, String> readAttributes(final Element element) {
        Map<String, String> map = newHashMap();
        NamedNodeMap xmlAttrs = element.getAttributes();
        for (int i = 0; i < xmlAttrs.getLength(); i++) {
            Attr item = (Attr) xmlAttrs.item(i);
            if (!RESERVED_ATTRIBUTE_NAMES.contains(item.getName())) {
                map.put(item.getName(), item.getValue());
            }
        }

        return map;
    }

    private void initMetadata() {
        ControlTask annotation = method.getAnnotation(ControlTask.class);
        description = annotation.description();
        label = annotation.label().equals("") ? name : annotation.label();
    }

    void verify(Verifications verifications) {
        if (method != null) {
            verifications.verify(descriptor.getType(), method.getParameterTypes().length == 0, "ControlTask %s should not take any parameters", getFqn());
            verifications.verify(descriptor.getType(), List.class.isAssignableFrom(method.getReturnType()), "ControlTask %s should return a List<Step>", getFqn());
        } else if (delegate != null) {
            verifications.verify(descriptor.getType(), DelegateRegistry.exists(delegate), "No delegate called [%s] available for control task [%s]", delegate, getFqn());
        } else if (!attributes.isEmpty()) {
            Method controlTaskDispatch = getExtendedDispatcher();
            verifications.verify(descriptor.getType(), controlTaskDispatch != null, "ControlTask dispatcher %s for %s is not present.", CONTROL_TASK_DISPATCH_METHOD, getFqn());
            verifications.verify(descriptor.getType(), List.class.isAssignableFrom(controlTaskDispatch.getReturnType()), "ControlTask dispatcher %s for %s should return a List<Step>", CONTROL_TASK_DISPATCH_METHOD, getFqn());
        } else {
            Method controlTaskDispatch = getExtendedDispatcher();
            if (controlTaskDispatch == null) {
                controlTaskDispatch = getSimpleDispatcher();
            }
            verifications.verify(descriptor.getType(), controlTaskDispatch != null, "ControlTask dispatcher %s for %s is not present.", CONTROL_TASK_DISPATCH_METHOD, getFqn());
            verifications.verify(descriptor.getType(), List.class.isAssignableFrom(controlTaskDispatch.getReturnType()), "ControlTask dispatcher %s for %s should return a List<Step>", CONTROL_TASK_DISPATCH_METHOD, getFqn());
        }
    }

    @Override
    public String getName() {
        return name;
    }

    @Override
    public String getLabel() {
        return label;
    }

    @Override
    public String getDescription() {
        return description;
    }

    @Override
    public Map<String, String> getAttributes() {
        return attributes;
    }

    @Override
    @SuppressWarnings("unchecked")
    public <T> T invoke(ConfigurationItem item) {
        try {
            if (method != null) {
                return (T) method.invoke(item);
            } else if (delegate != null) {
                return this.<T>invokeDelegate(item);
            } else {
                return this.<T>invokeDispatcher(item);
            }
        } catch (IllegalAccessException e) {
            throw new RuntimeException("Could not invoke " + name + " on " + item, e);
        } catch (InvocationTargetException e) {
            throw ReflectionUtils.handleInvocationTargetException(e, "Could not invoke " + name + " on " + item);
        }
    }

    @SuppressWarnings("unchecked")
    private <T> T invokeDelegate(final ConfigurationItem item) throws InvocationTargetException, IllegalAccessException {
        Method method = DelegateRegistry.getDelegate(delegate);
        Object o = DelegateRegistry.instantiateDelegate(delegate);
        return (T) method.invoke(o, item, name, attributes);
    }

    @SuppressWarnings("unchecked")
    private <T> T invokeDispatcher(final ConfigurationItem item) throws InvocationTargetException, IllegalAccessException {
        Method dispatcher = getExtendedDispatcher();
        if (dispatcher == null) {
            dispatcher = getSimpleDispatcher();
            return (T) dispatcher.invoke(item, name);
        }
        return (T) dispatcher.invoke(item, name, attributes);
    }

    @Override
    public String getFqn() {
        return String.format("%s.%s", descriptor.getType(), name);
    }

    private Method getSimpleDispatcher() {
        Class<?> clazz = descriptor.getClazz();
        try {
            return clazz.getMethod(CONTROL_TASK_DISPATCH_METHOD, String.class);
        } catch (NoSuchMethodException e) {
            return null;
        }
    }

    private Method getExtendedDispatcher() {
        Class<?> clazz = descriptor.getClazz();
        try {
            return clazz.getMethod(CONTROL_TASK_DISPATCH_METHOD, String.class, Map.class);
        } catch (NoSuchMethodException e) {
            return null;
        }
    }
}
