package com.xebialabs.deployit.booter.local;

import java.lang.annotation.Annotation;
import java.lang.reflect.Field;
import java.lang.reflect.Method;
import java.lang.reflect.Modifier;
import java.util.*;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;
import org.w3c.dom.Element;
import com.google.common.base.Preconditions;
import com.google.common.base.Predicate;
import com.google.common.base.Predicates;
import com.google.common.collect.Lists;
import com.google.common.collect.Maps;
import com.google.common.collect.Sets;

import com.xebialabs.deployit.plugin.api.reflect.*;
import com.xebialabs.deployit.plugin.api.udm.*;
import com.xebialabs.deployit.plugin.api.udm.artifact.Artifact;
import com.xebialabs.deployit.plugin.api.udm.artifact.DerivedArtifact;
import com.xebialabs.deployit.plugin.api.udm.artifact.SourceArtifact;
import com.xebialabs.deployit.plugin.api.udm.base.BaseConfigurationItem;
import com.xebialabs.deployit.plugin.api.validation.ValidationContext;
import com.xebialabs.deployit.plugin.api.validation.ValidationMessage;
import com.xebialabs.deployit.plugin.api.validation.Validator;

import static com.google.common.collect.Iterables.filter;
import static com.google.common.collect.Lists.newArrayList;
import static com.xebialabs.deployit.booter.local.SyntheticHelper.getOptionalBooleanAttribute;
import static com.xebialabs.deployit.booter.local.utils.ClassUtils.getActualTypeArguments;
import static com.xebialabs.deployit.booter.local.utils.ReflectionUtils.searchField;
import static com.xebialabs.deployit.plugin.api.Deprecations.deprecated;
import static java.lang.reflect.Modifier.isAbstract;


class LocalDescriptor implements Descriptor {

    static final String PLACEHOLDERS_FIELD = "placeholders";

    private Type type;
    private Class<? extends ConfigurationItem> clazz;
    private String description;
    private Metadata.ConfigurationItemRoot root;
    private List<Type> superclasses = Lists.newArrayList();
    private Set<Type> interfaces = Sets.newHashSet();
    private boolean virtual = false;

    private Map<String, PropertyDescriptor> properties = Maps.newLinkedHashMap();
    private Type deployableType;
    private Type containerType;

    private Type generatedDeployableType;
    private Type generatedDeployableBase;
    private String generatedDeployableDescription;
    private boolean generatedDeployableInheritsDefaultValues;

    private boolean hierarchyInitialized = false;
    private Map<String, MethodDescriptor> controlTasks = Maps.newHashMap();
    private List<Validator<ConfigurationItem>> validators = Lists.newArrayList();
    private transient List<TypeVerification> verifications = Lists.newArrayList();
    private Field syntheticPropertiesField;
    private boolean isInspectable;

    private LocalDescriptor(Class<? extends ConfigurationItem> clazz) {
        this.type = Type.valueOf(clazz);
        this.virtual = clazz.isInterface();
        this.clazz = clazz;
    }

    private LocalDescriptor(Type type) {
        this.type = type;
    }

    static Descriptor from(Class<? extends ConfigurationItem> clazz) {
        try {
            LocalDescriptor descriptor = new LocalDescriptor(clazz);
            descriptor.initMetadata();
            descriptor.scanClass();
            return descriptor;
        } catch (RuntimeException e) {
            throw new DescriptorException("Could not create descriptor for: " + clazz.getName(), e);
        } catch (Throwable t) {
            throw new DescriptorError("Could not create descriptor for: " + clazz.getName(), t);
        }
    }

    private void initMetadata() {
        if (virtual) {
            return;
        }

        boolean directlyAnnotated = false;
        Annotation[] declaredAnnotations = clazz.getDeclaredAnnotations();
        for (Annotation declaredAnnotation : declaredAnnotations) {
            if (declaredAnnotation.annotationType().equals(Metadata.class)) {
                directlyAnnotated = true;
            }
        }

        Metadata annotation = Preconditions.checkNotNull(clazz.getAnnotation(Metadata.class), "Class " + clazz.getName() + " or one of its ancestors does not have a @Metadata annotation");
        description = annotation.description();
        root = annotation.root();
        isInspectable = annotation.inspectable();
        virtual = (directlyAnnotated && annotation.virtual()) || Modifier.isAbstract(clazz.getModifiers());
    }

    private void scanClass() {
        findProperties();
        findControlTasks();
        findInterfaces();
        findValidations();
        findVerifications();

        Class<?> superclass = clazz.getSuperclass();
        if (superclass != null && ConfigurationItem.class.isAssignableFrom(superclass)) {
            Type supertype = Type.valueOf(superclass);
            addSuperClass(supertype);
        }

        initDeployableAndContainerTypes();
    }

    private void findVerifications() {
        for (Annotation annotation : clazz.getAnnotations()) {
            if (VerificationConverter.isVerification(annotation)) {
                this.verifications.add(VerificationConverter.makeVerification(annotation));
            }
        }
    }

    @SuppressWarnings("unchecked")
    private void findValidations() {
        for (Annotation annotation : clazz.getAnnotations()) {
            if (ValidationRuleConverter.isRule(annotation)) {
                Validator<ConfigurationItem> v = (Validator<ConfigurationItem>) ValidationRuleConverter.makeRule(annotation);
                this.validators.add(v);
            }
        }
    }

    private void findControlTasks() {
        for (Method method : clazz.getDeclaredMethods()) {
            if (method.isAnnotationPresent(ControlTask.class)) {
                MethodDescriptor from = LocalMethodDescriptor.from(this, method);
                addControlTask(from);
            }
        }
    }

    private void findProperties() {
        for (Field field : clazz.getDeclaredFields()) {
            if (field.isAnnotationPresent(Property.class)) {
                PropertyDescriptor propertyDescriptor = LocalPropertyDescriptor.from(this, field);
                addProperty(propertyDescriptor);
            }
        }
    }

    private void findInterfaces() {
        Class<?>[] clazzInterfaces = clazz.getInterfaces();
        List<Class<?>> allInterfacesFound = Lists.newArrayList();
        findAllSuperInterfaces(clazzInterfaces, allInterfacesFound);
        for (Class<?> clazzInterface : allInterfacesFound) {
            if (clazzInterface.getPackage().isAnnotationPresent(Prefix.class)) {
                addInterface(Type.valueOf(clazzInterface));
            }
        }
    }

    private void findAllSuperInterfaces(Class<?>[] childInterfaces, List<Class<?>> allInterfacesFound) {
        for (Class<?> childInterface : childInterfaces) {
            allInterfacesFound.add(childInterface);
            findAllSuperInterfaces(childInterface.getInterfaces(), allInterfacesFound);
        }
    }

    private void initDeployableAndContainerTypes() {
        if (Deployed.class.isAssignableFrom(clazz)) {
            @SuppressWarnings({"unchecked", "rawtypes"})
            List<Class<?>> typeArguments = getActualTypeArguments((Class<? extends Deployed>) clazz, Deployed.class);
            Preconditions.checkArgument(typeArguments.size() == 2, "Expected exactly a Deployable and a Container, but got %s", typeArguments);

            Class<?> deployableClass = typeArguments.get(0);
            if (deployableClass != null) {
                Preconditions.checkArgument(Deployable.class.isAssignableFrom(deployableClass), "Expected first item to be a deployable");
                deployableType = Type.valueOf(deployableClass);
            } else {
                deployableType = null;
            }

            Class<?> containerClass = typeArguments.get(1);
            if (containerClass != null) {
                Preconditions.checkArgument(Container.class.isAssignableFrom(containerClass), "Expected second item to be a container");
                containerType = Type.valueOf(containerClass);
            } else {
                containerType = null;
            }
        } else {
            deployableType = null;
            containerType = null;
        }
    }

    static Descriptor from(Element typeElement) {
        Type type = SyntheticHelper.getRequiredTypeAttribute(typeElement, "type");

        LocalDescriptor descriptor = new LocalDescriptor(type);
        descriptor.initSynthetic(typeElement);

        return descriptor;
    }

    private void initSynthetic(Element typeElement) {
        description = SyntheticHelper.getOptionalStringAttribute(typeElement, "description", "Description unavailable");
        virtual = getOptionalBooleanAttribute(typeElement, "virtual", false);
        isInspectable = getOptionalBooleanAttribute(typeElement, "inspectable", false);

        Type superType = SyntheticHelper.getRequiredTypeAttribute(typeElement, "extends");
        addSuperClass(superType);

        parseSyntheticDeployableAndContainerType(typeElement);
        parseProperties(typeElement);
        parseControlTasks(typeElement);
        parseValidators(typeElement);
        parseVerifications(typeElement);
    }

    private void parseVerifications(final Element typeElement) {
        SyntheticHelper.forEach(SyntheticHelper.childByName(typeElement, Predicates.equalTo(VerificationConverter.VERIFICATION_ELEMENT_NAME)), new SyntheticHelper.Closure<Element>() {
            @SuppressWarnings("unchecked")
            @Override
            public void call(Element element) {
                verifications.add(VerificationConverter.makeVerification(element, type));
            }
        });
    }

    private void parseValidators(Element typeElement) {
        SyntheticHelper.forEach(SyntheticHelper.childByName(typeElement, Predicates.equalTo(ValidationRuleConverter.RULE_ELEMENT_NAME)), new SyntheticHelper.Closure<Element>() {
            @SuppressWarnings("unchecked")
            @Override
            public void call(Element element) {
                validators.add((Validator<ConfigurationItem>) ValidationRuleConverter.makeRule(element, type));
            }
        });
    }

    private void parseSyntheticDeployableAndContainerType(Element typeElement) {
        deployableType = SyntheticHelper.getOptionalTypeAttribute(typeElement, "deployable-type");
        containerType = SyntheticHelper.getOptionalTypeAttribute(typeElement, "container-type");
        Iterator<Element> generateElements = SyntheticHelper.childByName(typeElement, Predicates.equalTo("generate-deployable"));
        if (generateElements.hasNext()) {
            Element generateDeployable = generateElements.next();
            generatedDeployableType = SyntheticHelper.getRequiredTypeAttribute(generateDeployable, "type");
            generatedDeployableBase = SyntheticHelper.getRequiredTypeAttribute(generateDeployable, "extends");
            generatedDeployableInheritsDefaultValues = getOptionalBooleanAttribute(generateDeployable, "copy-default-values", false);
            generatedDeployableDescription = SyntheticHelper.getOptionalStringAttribute(generateDeployable, "description", "Description unavailable");
        }
    }

    void parseTypeModification(Element element) {
        parseProperties(element);
        parseControlTasks(element);
        parseValidators(element);
        parseVerifications(element);
    }

    private void parseControlTasks(final Element element) {
        SyntheticHelper.forEach(SyntheticHelper.childByName(element, Predicates.equalTo("method")), new SyntheticHelper.Closure<Element>() {
            @Override
            public void call(Element element) {
                MethodDescriptor controlTask = LocalMethodDescriptor.from(LocalDescriptor.this, element);
                verifyNewControlTask(controlTask);
                addControlTask(controlTask);
            }
        });
    }

    private void parseProperties(final Element element) {
        SyntheticHelper.forEach(SyntheticHelper.childByName(element, Predicates.equalTo("property")), new SyntheticHelper.Closure<Element>() {
            @Override
            public void call(Element element) {
                String name = SyntheticHelper.getRequiredStringAttribute(element, "name");
                LocalPropertyDescriptor newDesc = LocalPropertyDescriptor.from(LocalDescriptor.this, element);
                PropertyDescriptor oldDesc = properties.get(name);
                if (oldDesc != null) {
                    newDesc.overrideWith((LocalPropertyDescriptor) oldDesc);
                }
                addProperty(newDesc);
            }
        });
    }

    private void verifyNewControlTask(MethodDescriptor controlTask) {
        if (controlTasks.containsKey(controlTask.getName())) {
            throw new IllegalStateException("Cannot override existing Control Task [" + controlTask.getFqn() + "] with a synthetic one.");
        }
    }

    static Descriptor from(LocalDescriptor deployedDescriptor) {
        LocalDescriptor deployableDescriptor = new LocalDescriptor(deployedDescriptor.generatedDeployableType);
        deployableDescriptor.addSuperClass(deployedDescriptor.generatedDeployableBase);
        deployableDescriptor.initDeployableFromDeployed(deployedDescriptor, deployedDescriptor.generatedDeployableDescription);
        return deployableDescriptor;
    }

    private void initDeployableFromDeployed(LocalDescriptor deployedDescriptor, String generatedDeployableDescription) {
        if (generatedDeployableDescription == null || generatedDeployableDescription.equals("Description unavailable")) {
            description = deployedDescriptor.getDescription() + " (deployable)";
        } else {
            description = generatedDeployableDescription;
        }

        for (PropertyDescriptor deployedPropertyDescriptor : deployedDescriptor.getPropertyDescriptors()) {
            String name = deployedPropertyDescriptor.getName();
            boolean isUdmField = name.equals(Deployed.DEPLOYABLE_FIELD) || name.equals(Deployed.CONTAINER_FIELD)
                    || name.equals(PLACEHOLDERS_FIELD);
            final PropertyKind kind = deployedPropertyDescriptor.getKind();
            boolean isReferenceField = kind == PropertyKind.CI || kind == PropertyKind.SET_OF_CI || kind == PropertyKind.LIST_OF_CI;
            if (isUdmField || isReferenceField || deployedPropertyDescriptor.isHidden()) {
                continue;
            }

            PropertyDescriptor generatedDeployablePropertyDescriptor = LocalPropertyDescriptor.generateDeployableFrom(this, deployedPropertyDescriptor);
            if (deployedDescriptor.generatedDeployableInheritsDefaultValues) {
                GlobalContext.register(generatedDeployablePropertyDescriptor, GlobalContext.lookup(deployedPropertyDescriptor));
            }
            addProperty(generatedDeployablePropertyDescriptor);
        }
    }

    void initHierarchy() {
        if (hierarchyInitialized) {
            return;
        }

        if (!superclasses.isEmpty()) {
            Type toInitFrom = superclasses.get(0);

            do {
                LocalDescriptor superDesc = (LocalDescriptor) LocalDescriptorRegistry.getDescriptor(toInitFrom);
                if (superDesc == null) {
                    throw new IllegalStateException("Cannot build type hierarchy for " + getType() + " because one of its supertypes cannot be found: " + toInitFrom + " not found");
                }

                // For synthetic types, root and clazz might not be set yet.
                if (root == null) {
                    root = superDesc.getRoot();
                }

                if (clazz == null) {
                    clazz = superDesc.clazz;
                }

                inheritPropertyDescriptors(properties, superDesc.properties);
                inheritControlTasks(controlTasks, superDesc.controlTasks);
                inheritValidators(validators, superDesc.validators);

                for (Type superclass : superDesc.superclasses) {
                    addSuperClass(superclass);
                }
                for (Type intf : superDesc.interfaces) {
                    addInterface(intf);
                }
                toInitFrom = superDesc.superclasses.isEmpty() || superDesc.hierarchyInitialized ? null : superDesc.getSuperClasses().get(0);

                if (deployableType == null) {
                    deployableType = superDesc.getDeployableType();
                }

                if (containerType == null) {
                    containerType = superDesc.getContainerType();
                }
            } while (toInitFrom != null);
        }

        if (!clazz.isInterface() && !Type.valueOf("api.ValidatedConfigurationItem").equals(type)) {
            syntheticPropertiesField = searchField(clazz, ConfigurationItem.SYNTHETIC_PROPERTIES_FIELD);
            syntheticPropertiesField.setAccessible(true);
        }

        hierarchyInitialized = true;
    }

    void verify(final Verifications verifications) {
        verifySyntheticPropertiesField(verifications);
        verifyReferenceTypes(verifications);
        verifyNonArtifactDoesNotHavePlaceholders(verifications);
        verifyArtifactInterfaces(verifications);
        if (!isVirtual()) {
            verifyJavaClassIsNotAbstract(verifications);
            verifyDeployedHasDeployableAndContainerType(verifications);
            verifyDeployedHasCorrectInheritance(verifications);
            verifyHiddenRequiredPropertiesHaveDefaultValue(verifications);
            verifyControlTasks(verifications);
        }
        if (!isInspectable()) {
            verifyNoInspectionProperties(verifications);
        }
        verifyExtendsBaseConfigurationItem(verifications);

        for (TypeVerification verification : this.verifications) {
            verification.verify(this, new VerificationContext() {
                @Override
                public void error(final String message, final Object... params) {
                    verifications.verify(type, false, message, params);
                }
            });
        }

        for (PropertyDescriptor propertyDescriptor : properties.values()) {
            ((LocalPropertyDescriptor) propertyDescriptor).verify(verifications);
        }
    }

    private void verifyExtendsBaseConfigurationItem(final Verifications verifications) {
        if (clazz != null) {
            if (!BaseConfigurationItem.class.isAssignableFrom(clazz)) {
                deprecated("ConfigurationItem [%s] does not extend BaseConfigurationItem, will be required in Deployit 4.0", type);
                logger.warn("Concurrent modifications cannot be detected on Configuration Items of type [{}] because is does not extend BaseConfigurationItem.", type);
            }
        }
    }

    private void verifyNoInspectionProperties(Verifications verifications) {
        List<PropertyDescriptor> inspectableProperties = newArrayList(filter(properties.values(), new Predicate<PropertyDescriptor>() {
            @Override
            public boolean apply(PropertyDescriptor input) {
                return input.isInspectionProperty();
            }
        }));
        verifications.verify(type, inspectableProperties.isEmpty(), "Type [%s] is not inspectable but has @InspectionProperty properties %s", type, inspectableProperties);
    }

    private void verifySyntheticPropertiesField(Verifications verifications) {
        verifications.verify(type, syntheticPropertiesField != null, "Synthetic properties field should be set");
        if (syntheticPropertiesField != null) {
            verifications.verify(type, syntheticPropertiesField.getType().isAssignableFrom(Map.class), "Synthetic properties field should be Map<String, Object>, not: %s", syntheticPropertiesField.getType());
        }
    }


    private void verifyControlTasks(Verifications verifications) {
        for (MethodDescriptor controlTask : controlTasks.values()) {
            ((LocalMethodDescriptor) controlTask).verify(verifications);
        }
    }

    private void verifyReferenceTypes(Verifications verifications) {
        for (PropertyDescriptor p : properties.values()) {
            final PropertyKind kind = p.getKind();
            if (kind == PropertyKind.CI || kind == PropertyKind.SET_OF_CI || kind == PropertyKind.LIST_OF_CI) {
                verifications.verify(type, isValidReferencedType(p.getReferencedType()), "Property %s of type %s refers to non-existing type %s", p.getName(), ((LocalPropertyDescriptor) p).getDeclaringDescriptor().getType(), p.getReferencedType());
            }
        }
    }

    private boolean isValidReferencedType(Type referencedType) {
        if (LocalDescriptorRegistry.exists(referencedType))
            return true;

        for (Descriptor d : LocalDescriptorRegistry.getDescriptors()) {
            if (d.getSuperClasses().contains(referencedType)) {
                return true;
            }
            if (d.getInterfaces().contains(referencedType)) {
                return true;
            }
        }

        return false;
    }

    private void verifyNonArtifactDoesNotHavePlaceholders(Verifications verifications) {
        if (!isAssignableTo(Artifact.class)) {
            for (PropertyDescriptor propertyDescriptor : properties.values()) {
                verifications.verify(type, !propertyDescriptor.getName().equals(PLACEHOLDERS_FIELD), "Cannot have a field 'placeholders' as 'udm.Artifact' is not implemented.");
            }
        }
    }

    private void verifyArtifactInterfaces(Verifications verifications) {
        if (isAssignableTo(Deployable.class) && isAssignableTo(Artifact.class)) {
            verifications.verify(type, isAssignableTo(SourceArtifact.class), "Implements both 'udm.Deployable' and 'udm.Artifact' interface. Must also implement the 'udm.SourceArtifact' interface");
            verifications.verify(type, isAssignableTo(DeployableArtifact.class), "Implements the 'udm.Deployable' and 'udm.Artifact' interface. Must also implement the 'udm.DeployableArtifact' interface");
        }
        if (isAssignableTo(Deployed.class) && isAssignableTo(Artifact.class)) {
            verifications.verify(type, isAssignableTo(DerivedArtifact.class), "Implements the 'udm.Deployed' and 'udm.Artifact' interface. Must also implement the 'udm.DerivedArtifact' interface");
        }
    }

    private void verifyJavaClassIsNotAbstract(Verifications verifications) {
        verifications.verify(type, !isAbstract(clazz.getModifiers()), "Non-virtual type %s has an abstract Java class %s", type, clazz.getName());
    }

    private void verifyDeployedHasDeployableAndContainerType(Verifications verifications) {
        Type deployedType = Type.valueOf(Deployed.class);
        if (isAssignableTo(deployedType)) {
            verifications.verify(type, getDeployableType() != null, "Non-virtual type %s is a sub-type of %s but does not have a deployable-type", getType(), deployedType);
            verifications.verify(type, getContainerType() != null, "Non-virtual type %s is a sub-type of %s but does not have a container-type", getType(), deployedType);
        }
    }

    private void verifyDeployedHasCorrectInheritance(Verifications verifications) {
        Type deployedType = Type.valueOf(Deployed.class);
        if (isAssignableTo(deployedType)) {
            Type directSuperType = superclasses.get(0);
            Descriptor superDescriptor = directSuperType.getDescriptor();
            Type superDeployable = superDescriptor.getDeployableType();
            Type superContainer = superDescriptor.getContainerType();
            verifications.verify(type, superDeployable == null || deployableType.instanceOf(superDeployable), "Type %s inherits from %s<%s, %s> but doesn't have a deployable subtype (%s).", type, directSuperType, superDeployable, superContainer, deployableType);
            verifications.verify(type, superContainer == null || containerType.instanceOf(superContainer), "Type %s inherits from %s<%s, %s> but doesn't have a container subtype (%s).", type, directSuperType, superDeployable, superContainer, containerType);
        }
    }

    private void verifyHiddenRequiredPropertiesHaveDefaultValue(Verifications verifications) {
        for (PropertyDescriptor p : properties.values()) {
            if (p.isHidden() && p.isRequired()) {
                verifications.verify(type, p.getDefaultValue() != null, "Hidden required property %s of non-virtual type %s must have a default value", p.getName(), getType());
            }
        }
    }

    private void inheritValidators(List<Validator<ConfigurationItem>> dest, List<Validator<ConfigurationItem>> source) {
        dest.addAll(source);
    }

    private void inheritControlTasks(Map<String, MethodDescriptor> dest, Map<String, MethodDescriptor> source) {
        for (Map.Entry<String, MethodDescriptor> sourceEntry : source.entrySet()) {
            if (!dest.containsKey(sourceEntry.getKey())) {
                dest.put(sourceEntry.getKey(), new LocalMethodDescriptor((LocalMethodDescriptor) sourceEntry.getValue(), this));
            } else {
                logger.warn("Not inheriting ControlTask [{}] on [{}]", new Object[]{sourceEntry.getValue().getFqn(), type});
            }
        }
    }

    private void inheritPropertyDescriptors(Map<String, PropertyDescriptor> dest, Map<String, PropertyDescriptor> source) {
        Map<String, PropertyDescriptor> myDescriptors = Maps.newLinkedHashMap(dest);
        dest.clear();
        for (Map.Entry<String, PropertyDescriptor> sourceEntry : source.entrySet()) {
            if (!myDescriptors.containsKey(sourceEntry.getKey())) {
                dest.put(sourceEntry.getKey(), new LocalPropertyDescriptor((LocalPropertyDescriptor) sourceEntry.getValue(), this));
            } else {
                ((LocalPropertyDescriptor) myDescriptors.get(sourceEntry.getKey())).overrideWith((LocalPropertyDescriptor) sourceEntry.getValue());
            }
        }
        dest.putAll(myDescriptors);
    }

    private void addSuperClass(Type supertype) {
        superclasses.add(supertype);
        LocalDescriptorRegistry.registerSubtype(supertype, type);
    }

    private void addInterface(Type intf) {
        LocalDescriptorRegistry.registerSubtype(intf, type);
        interfaces.add(intf);
    }

    private void addProperty(PropertyDescriptor propertyDescriptor) {
        properties.put(propertyDescriptor.getName(), propertyDescriptor);
    }

    private void addControlTask(MethodDescriptor from) {
        controlTasks.put(from.getName(), from);
    }

    @Override
    public Type getType() {
        return type;
    }

    @Override
    public Class<?> getClazz() {
        return clazz;
    }

    @Override
    public String getDescription() {
        return description;
    }

    @Override
    public Metadata.ConfigurationItemRoot getRoot() {
        return root;
    }

    @Override
    public Collection<PropertyDescriptor> getPropertyDescriptors() {
        return properties.values();
    }

    @Override
    public PropertyDescriptor getPropertyDescriptor(String name) {
        return properties.get(name);
    }

    @Override
    public MethodDescriptor getControlTask(String name) {
        return controlTasks.get(name);
    }

    @Override
    public Collection<MethodDescriptor> getControlTasks() {
        return controlTasks.values();
    }

    @Override
    public boolean isAssignableTo(Class<?> clazz) {
        return isAssignableTo(Type.valueOf(clazz));
    }

    @Override
    public boolean isAssignableTo(Type type) {
        return this.type.isSubTypeOf(type) || this.type.equals(type);
    }

    @Override
    public List<Type> getSuperClasses() {
        return superclasses;
    }

    @Override
    public Set<Type> getInterfaces() {
        return interfaces;
    }

    @Override
    public boolean isVirtual() {
        return virtual;
    }

    @Override
    public boolean areEqual(ConfigurationItem item, ConfigurationItem other) {
        return areEqual(item, other, new HashSet<String>());
    }

    boolean areEqual(ConfigurationItem item, ConfigurationItem other, Set<String> itemsBeingCompared) {
        if (item == null) {
            return other == null;
        }

        if (!getType().equals(item.getType()) || !getType().equals(other.getType())) {
            return false;
        }

        if (!itemsBeingCompared.add(item.getId())) {
            return true;
        }

        for (PropertyDescriptor pd : getPropertyDescriptors()) {
            if (!((LocalPropertyDescriptor) pd).areEqual(item, other, itemsBeingCompared)) {
                return false;
            }
        }
        return true;
    }

    boolean shouldGenerateDeployableType() {
        return generatedDeployableType != null;
    }

    @Override
    @SuppressWarnings("unchecked")
    public <T extends ConfigurationItem> T newInstance() {
        if (virtual) {
            throw new IllegalArgumentException("Cannot instantiate class for " + type + " because it is virtual");
        }

        try {
            Field typeField = searchField(clazz, ConfigurationItem.TYPE_FIELD);
            typeField.setAccessible(true);
            T t = (T) clazz.newInstance();
            typeField.set(t, type);

            prefillDefaultProperties(t);

            return t;
        } catch (InstantiationException exc) {
            throw new RuntimeException("Cannot instantiate class " + clazz.getName(), exc);
        } catch (IllegalAccessException exc) {
            throw new RuntimeException("Cannot instantiate class " + clazz.getName(), exc);
        }
    }

    private <T extends ConfigurationItem> void prefillDefaultProperties(T t) {
        for (PropertyDescriptor pd : getPropertyDescriptors()) {
            LocalPropertyDescriptor lpd = (LocalPropertyDescriptor) pd;
            if (pd.getDefaultValue() != null) {
                pd.set(t, lpd.getDefaultValue());
            } else {
                pd.set(t, lpd.emptyValue());
            }
        }
    }

    @Override
    public String toString() {
        return "Descriptor[" + type + "]";
    }

    @Override
    public Type getDeployableType() {
        return deployableType;
    }

    @Override
    public Type getContainerType() {
        return containerType;
    }

    Field getSyntheticPropertiesField() {
        return syntheticPropertiesField;
    }

    @Override
    public List<ValidationMessage> validate(final ConfigurationItem ci) {
        final ArrayList<ValidationMessage> messages = Lists.newArrayList();
        for (PropertyDescriptor propertyDescriptor : properties.values()) {
            ((LocalPropertyDescriptor) propertyDescriptor).validate(ci, messages);
        }
        for (Validator<ConfigurationItem> validator : validators) {
            validator.validate(ci, new ValidationContext() {
                @Override
                public void error(String message, Object... params) {
                    messages.add(new ValidationMessage(ci.getId(), null, String.format(message, params)));
                }
            });
        }
        return messages;
    }

    @Override
    public boolean isInspectable() {
        return isInspectable;
    }

    @SuppressWarnings("serial")
    private static class DescriptorException extends RuntimeException {
        public DescriptorException(String s, RuntimeException e) {
            super(s, e);
        }
    }

    private static class DescriptorError extends Error {
        public DescriptorError(String s, Throwable e) {
            super(s, e);
        }
    }

    private static final Logger logger = LoggerFactory.getLogger(LocalDescriptor.class);
}
