package com.xebialabs.deployit.booter.local;

import java.lang.reflect.Method;
import java.util.List;
import java.util.Map;
import java.util.Set;

import com.xebialabs.deployit.plugin.api.udm.ConfigurationItem;
import com.xebialabs.deployit.plugin.api.udm.Delegate;

import nl.javadude.scannit.Scannit;

import static com.google.common.collect.Lists.newArrayList;
import static com.google.common.collect.Maps.newHashMap;

public class DelegateRegistry {
    private static final Map<String, Method> DELEGATES = newHashMap();

    static void boot() {
        Set<Method> delegateMethods = Scannit.getInstance().getMethodsAnnotatedWith(Delegate.class);
        for (Method delegateMethod : delegateMethods) {
            Delegate annotation = delegateMethod.getAnnotation(Delegate.class);
            Method put = DELEGATES.put(annotation.name(), delegateMethod);
            if (put != null) {
                throw new IllegalStateException(String.format("Found 2 methods with same delegate name [%s]: %s and %s", annotation.name(), put, delegateMethod));
            }
        }
    }

    static void verify(Verifications verifications) {
        for (Map.Entry<String, Method> entry : DELEGATES.entrySet()) {
            Method method = entry.getValue();
            String name = entry.getKey();
            verifications.verify(List.class.isAssignableFrom(method.getReturnType()), "Delegate [%s (%s)] should have a List<Step> return type", name, method);
            Class<?>[] parameterTypes = method.getParameterTypes();
            verifications.verify(parameterTypes.length == 3, "Delegate [%s (%s)] should take 3 arguments (Taking: %s)", name, method, newArrayList(parameterTypes));
            verifications.verify(ConfigurationItem.class.isAssignableFrom(parameterTypes[0]), "Delegate [%s (%s)] should take a ConfigurationItem as first parameter", name, method);
            verifications.verify(parameterTypes[1].equals(String.class), "Delegate [%s (%s)] should take a String as second parameter", name, method);
            verifications.verify(Map.class.isAssignableFrom(parameterTypes[2]), "Delegate [%s (%s)] should take a Map<String, String> as third parameter", name, method);
            Class<?> declaringClass = method.getDeclaringClass();
            try {
                declaringClass.getConstructor();
            } catch (NoSuchMethodException e) {
                verifications.verify(false, "Delegate [%s (%s)] does not have a public no-arg constructor.", name, method);
            }
        }
    }

    static boolean exists(String name) {
        return DELEGATES.containsKey(name);
    }

    static Method getDelegate(String name) {
        return DELEGATES.get(name);
    }

    static Object instantiateDelegate(String name) {
        try {
            return DELEGATES.get(name).getDeclaringClass().newInstance();
        } catch (InstantiationException e) {
            throw new RuntimeException("Could not instantiate delegate: " + name, e);
        } catch (IllegalAccessException e) {
            throw new RuntimeException("Could not instantiate delegate: " + name, e);
        }
    }
}
