package com.xebialabs.deployit.booter.local;

import java.util.Collection;
import java.util.List;
import java.util.Map;
import java.util.Stack;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;
import org.w3c.dom.Element;
import com.google.common.base.Function;
import com.google.common.base.Predicate;

import com.xebialabs.deployit.plugin.api.reflect.Type;
import com.xebialabs.deployit.plugin.api.udm.ConfigurationItem;
import com.xebialabs.deployit.plugin.api.udm.Parameters;

import static com.google.common.base.Preconditions.checkNotNull;
import static com.google.common.base.Preconditions.checkState;
import static com.google.common.collect.Collections2.filter;
import static com.google.common.collect.Collections2.transform;
import static com.google.common.collect.Lists.newArrayList;
import static com.google.common.collect.Maps.newHashMap;
import static com.xebialabs.deployit.booter.local.LocalDescriptor.*;
import static com.xebialabs.deployit.booter.local.utils.ReflectionUtils.getAllInterfaces;
import static com.xebialabs.deployit.booter.local.utils.XmlUtils.getRequiredStringAttribute;
import static com.xebialabs.deployit.booter.local.utils.XmlUtils.getRequiredTypeAttribute;
import static java.lang.String.format;

class TypeDefinitions {
    private final Map<Type, TypeDefinition> typeDefs = newHashMap();

    void defineType(Class clazz) {
        ClassBasedTypeDefinition typeDef = new ClassBasedTypeDefinition(clazz);
        registerTypeDefinition(typeDef);
        logger.debug("Found: {}", typeDef);
    }

    protected void registerTypeDefinition(TypeDefinition typeDef) {
        checkState(!typeDefs.containsKey(typeDef.type), "Trying to register duplicate definition for type [%s]. Existing: [%s], duplicate: [%s]", typeDef.type, typeDefs.get(typeDef.type), typeDef);
        typeDefs.put(typeDef.type, typeDef);
    }

    void defineType(Element element) {
        SyntheticBasedTypeDefinition typeDef = new SyntheticBasedTypeDefinition(element);
        registerTypeDefinition(typeDef);
        logger.debug("Found: {}", typeDef);

    }

    void defineGeneratedType(Element element, Element owner) {
        GeneratedTypeDefinition typeDef = new GeneratedTypeDefinition(owner, element);
        registerTypeDefinition(typeDef);
        logger.debug("Found: {}", typeDef);
    }

    void defineGeneratedParameterType(final Element methodDef, final Element type) {
        GeneratedParameterTypeDefinition typeDef = new GeneratedParameterTypeDefinition(type, methodDef);
        registerTypeDefinition(typeDef);
        logger.debug("Found: {}", typeDef);
    }

    void modifyType(Element element) {
        TypeMod typeMod = new TypeMod(getRequiredTypeAttribute(element, "type"), element);
        checkState(typeDefs.containsKey(typeMod.type), "Detected type-modification for non-existing type [%s]", typeMod.type);
        typeDefs.get(typeMod.type).typeModifications.push(typeMod);
        logger.debug("Found: {}", typeMod);
    }

    Collection<TypeDefinition> getDefinitions() {
        return newArrayList(typeDefs.values());
    }

    TypeDefinition getDefinition(Type type) {
        return checkNotNull(typeDefs.get(type), "Could not find a type definition associated with type [%s]", type);
    }

    private static Type type(Class clazz) {
        if (clazz != null && ConfigurationItem.class.isAssignableFrom(clazz)) {
            return Type.valueOf(clazz);
        } else {
            return null;
        }
    }

    public static Type generatedParameterType(final Type type, final String name) {
        return Type.valueOf(type.toString() + "_" + name);
    }

    class TypeMod {
        private Type type;
        private Element modification;

        protected TypeMod(final Type type, final Element modification) {
            this.type = type;
            this.modification = modification;
        }

        @Override
        public String toString() {
            return this.getClass().getSimpleName() + "[" + type + "]";
        }
    }

    abstract class TypeDefinition {
        Type type;
        Type parent;
        Stack<TypeMod> typeModifications = new Stack<TypeMod>();

        @Override
        public String toString() {
            return this.getClass().getSimpleName() + "[" + type + "]";
        }

        final void register(TypeDefinitions typeDefinitions) {
            if (LocalDescriptorRegistry.exists(type)) {
                return;
            }

            if (parent != null && !LocalDescriptorRegistry.exists(parent)) {
                typeDefinitions.getDefinition(parent).register(typeDefinitions);
            }

            LocalDescriptor descriptor = createDescriptor(typeDefinitions);
            LocalDescriptorRegistry.register(descriptor);

            applyTypeModifications(descriptor);
        }

        protected abstract LocalDescriptor createDescriptor(final TypeDefinitions typeDefinitions);

        void applyTypeModifications(LocalDescriptor descriptor) {
            for (TypeMod typeModification : typeModifications) {
                descriptor.parseTypeModification(typeModification.modification);
            }
        }

        final void registerTypeTree() {
            registerAsSubtype(type);
        }

        protected void registerAsSubtype(Type type) {
            if (this.parent != null) {
                LocalDescriptorRegistry.registerSubtype(parent, type);
                getDefinition(this.parent).registerAsSubtype(type);
            }
        }
    }

    class ClassBasedTypeDefinition extends TypeDefinition {
        private Class<? extends ConfigurationItem> clazz;
        private List<Type> interfaces = newArrayList();

        public ClassBasedTypeDefinition(Class<? extends ConfigurationItem> ci) {
            this.type = type(ci);
            this.parent = type(ci.getSuperclass());

            interfaces = newArrayList(transform(filter(getAllInterfaces(ci), new Predicate<Class<?>>() {
                public boolean apply(Class<?> input) {
                    return type(input) != null;
                }
            }), new Function<Class, Type>() {
                public Type apply(Class input) {
                    return type(input);
                }
            }));
            this.clazz = ci;
        }


        @Override
        protected LocalDescriptor createDescriptor(final TypeDefinitions typeDefinitions) {
            for (Type anInterface : interfaces) {
                if (!LocalDescriptorRegistry.exists(anInterface)) {
                    typeDefinitions.getDefinition(anInterface).register(typeDefinitions);
                }
            }

            return from(clazz);
        }

        @Override
        protected void registerAsSubtype(Type type) {
            super.registerAsSubtype(type);
            for (Type anInterface : interfaces) {
                LocalDescriptorRegistry.registerSubtype(anInterface, type);
            }
        }
    }

    class SyntheticBasedTypeDefinition extends TypeDefinition {
        private Element element;

        public SyntheticBasedTypeDefinition(Element element) {
            this.element = element;
            this.type = getRequiredTypeAttribute(element, "type");
            this.parent = getRequiredTypeAttribute(element, "extends");
        }

        @Override
        protected LocalDescriptor createDescriptor(final TypeDefinitions typeDefinitions) {
            return fromSynthetic(element);
        }
    }

    class GeneratedTypeDefinition extends SyntheticBasedTypeDefinition {
        Type owner;

        private GeneratedTypeDefinition(Element owner, Element generatedElement) {
            super(generatedElement);
            this.owner = getRequiredTypeAttribute(owner, "type");
        }

        @Override
        protected LocalDescriptor createDescriptor(final TypeDefinitions typeDefinitions) {
            if (!LocalDescriptorRegistry.exists(owner)) {
                typeDefinitions.getDefinition(owner).register(typeDefinitions);
            }

            return fromGenerateDeployable((LocalDescriptor) LocalDescriptorRegistry.getDescriptor(owner));
        }
    }

    class GeneratedParameterTypeDefinition extends TypeDefinition {

        private final Type owner;
        private final Element methodDef;

        public GeneratedParameterTypeDefinition(final Element owner, final Element methodDef) {
            this.owner = getRequiredTypeAttribute(owner, "type");
            this.methodDef = methodDef;
            String methodName = getRequiredStringAttribute(methodDef, "name");
            this.type = generatedParameterType(this.owner, methodName);
            this.parent = Type.valueOf(Parameters.class);
        }

        @Override
        protected LocalDescriptor createDescriptor(final TypeDefinitions typeDefinitions) {
            if (!LocalDescriptorRegistry.exists(owner)) {
                typeDefinitions.getDefinition(owner).register(typeDefinitions);
            }

            return LocalDescriptor.fromGeneratedParameters(this.type, methodDef);
        }
    }

    private static final Logger logger = LoggerFactory.getLogger(TypeDefinitions.class);
}
