package com.xebialabs.deployit.booter.local;

import java.lang.annotation.Annotation;
import java.lang.reflect.Field;
import java.lang.reflect.ParameterizedType;
import java.util.*;
import javax.xml.bind.DatatypeConverter;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;
import org.w3c.dom.Element;
import com.google.common.base.Function;
import com.google.common.base.Predicates;
import com.google.common.base.Splitter;
import com.google.common.collect.*;

import com.xebialabs.deployit.booter.local.utils.ReflectionUtils;
import com.xebialabs.deployit.booter.local.validation.CollectionTypeValidator;
import com.xebialabs.deployit.booter.local.validation.ReferenceCollectionTypeValidator;
import com.xebialabs.deployit.booter.local.validation.ReferenceTypeValidator;
import com.xebialabs.deployit.booter.local.validation.RequiredValidator;
import com.xebialabs.deployit.plugin.api.inspection.InspectionProperty;
import com.xebialabs.deployit.plugin.api.reflect.Descriptor;
import com.xebialabs.deployit.plugin.api.reflect.PropertyDescriptor;
import com.xebialabs.deployit.plugin.api.reflect.PropertyKind;
import com.xebialabs.deployit.plugin.api.reflect.Type;
import com.xebialabs.deployit.plugin.api.udm.ConfigurationItem;
import com.xebialabs.deployit.plugin.api.udm.Property;
import com.xebialabs.deployit.plugin.api.validation.ValidationContext;
import com.xebialabs.deployit.plugin.api.validation.ValidationMessage;
import com.xebialabs.deployit.plugin.api.validation.Validator;

import static com.google.common.base.Preconditions.checkArgument;
import static com.google.common.base.Strings.emptyToNull;
import static com.google.common.collect.Lists.newArrayList;
import static com.google.common.collect.Maps.newHashMap;
import static com.google.common.collect.Maps.newLinkedHashMap;
import static com.google.common.collect.Sets.*;
import static com.xebialabs.deployit.booter.local.utils.Strings.deCamelize;
import static com.xebialabs.deployit.booter.local.utils.Strings.isBlank;
import static com.xebialabs.deployit.booter.local.utils.XmlUtils.*;
import static com.xebialabs.deployit.plugin.api.reflect.PropertyKind.*;
import static java.lang.String.format;
import static java.util.EnumSet.of;

class LocalPropertyDescriptor implements PropertyDescriptor {

    private LocalDescriptor declaringDescriptor;
    private String name;

    private Field field;
    private boolean asContainment;
    private String category;
    private String description;
    private String label;
    private boolean password;
    private boolean required;
    private Property.Size size;
    private PropertyKind kind;
    private List<String> enumValues;
    private Class<?> enumClass;
    private Type referencedType;
    private boolean hidden;
    private boolean inspectionProperty;
    private boolean requiredForInspection;
    private boolean isTransient;
    private Set<Validator<?>> validationRules = newHashSet();
    private Set<String> aliases = newHashSet();

    private LocalPropertyDescriptor(Field field) {
        this.name = field.getName();
        this.field = field;
        field.setAccessible(true);
    }

    private LocalPropertyDescriptor(String name) {
        this.name = name;
        this.field = null;
    }

    LocalPropertyDescriptor(LocalPropertyDescriptor pd, LocalDescriptor newOwner) {
        this.declaringDescriptor = newOwner;
        this.name = pd.name;
        this.field = pd.field;
        this.asContainment = pd.asContainment;
        this.category = pd.category;
        this.description = pd.description;
        this.label = pd.label;
        this.password = pd.password;
        this.required = pd.required;
        this.size = pd.size;
        this.kind = pd.kind;
        this.enumValues = pd.enumValues;
        this.enumClass = pd.enumClass;
        this.referencedType = pd.referencedType;
        this.hidden = pd.hidden;
        this.inspectionProperty = pd.inspectionProperty;
        this.requiredForInspection = pd.requiredForInspection;
        this.validationRules = newHashSet(pd.validationRules);
        this.aliases = newHashSet(pd.aliases);
        this.isTransient = pd.isTransient();
        // Clone the default value...
        GlobalContext.register(this, pd);
    }

    static PropertyDescriptor from(LocalDescriptor descriptor, Field field) {
        Property annotation = field.getAnnotation(Property.class);

        LocalPropertyDescriptor propertyDescriptor = new LocalPropertyDescriptor(field);
        propertyDescriptor.declaringDescriptor = descriptor;
        propertyDescriptor.initMetadata(annotation);
        propertyDescriptor.initType(annotation);
        initInspectionMetadata(field, propertyDescriptor);
        initValidationMetadata(field, propertyDescriptor);
        return propertyDescriptor;
    }

    private static void initValidationMetadata(Field field, LocalPropertyDescriptor propertyDescriptor) {
        propertyDescriptor.addDefaultValidationRules();
        for (Annotation annotation : field.getAnnotations()) {
            if (ValidationRuleConverter.isRule(annotation)) {
                propertyDescriptor.validationRules.add(ValidationRuleConverter.makeRule(annotation, propertyDescriptor));
            }
        }
    }

    static LocalPropertyDescriptor from(LocalDescriptor descriptor, Element propertyElement) {
        String name = getRequiredStringAttribute(propertyElement, "name");

        LocalPropertyDescriptor propertyDescriptor = new LocalPropertyDescriptor(name);
        propertyDescriptor.declaringDescriptor = descriptor;
        propertyDescriptor.initSynthetic(propertyElement);
        return propertyDescriptor;
    }

    public void verify(Verifications verifications) {
        verifyName(verifications, "name");
        // UI naming convention.
        verifyName(verifications, "displayName");
        verifyName(verifications, "id");
        verifyName(verifications, "type");
        verifications.verify(!getName().contains("$"), "Cannot define a property named '%s' because it contains a '$'", getName());
        verifications.verify(!hidden || isTransient, "Hidden property '%s' should be transient", this);
 
        if (kind == PropertyKind.BOOLEAN && required) {
            logger.warn("Required boolean property [{}] will be treated as an optional property.", this);
            required = false;
        }

        if (kind == PropertyKind.INTEGER && field != null && field.getType() == int.class) {
            if (!required) {
                logger.warn("Optional integer property [{}] backed by a Java int field will be treated as a required property.", this);
                required = true;
                validationRules.add(new RequiredValidator());
            }
        }

        verifications.verify(!asContainment || of(LIST_OF_CI, SET_OF_CI, CI).contains(kind), "'%s' can only be an as containment relation if it is a 'set_of_ci', 'list_of_ci' or 'ci' kind property", this);
        verifications.verify(referencedType == null || of(LIST_OF_CI, SET_OF_CI, CI).contains(kind), "'%s' can only have reference type '%s' set when it is a (Set/List of) CI kind property", this, referencedType);
        verifyAliases(verifications);
    }

    private void verifyAliases(Verifications verifications) {
        if (!aliases.isEmpty()) {
            for (PropertyDescriptor otherPd : declaringDescriptor.getPropertyDescriptors()) {
                LocalPropertyDescriptor pd = (LocalPropertyDescriptor) otherPd;
                if (!otherPd.equals(this)) {
                    verifications.verify(declaringDescriptor.getType(), !aliases.contains(pd.getName()), "Aliases of [%s] contain name [%s] which is an existing property.", this, pd.getName());
                    ImmutableSet<String> intersection = Sets.intersection(aliases, pd.aliases).immutableCopy();
                    verifications.verify(declaringDescriptor.getType(), intersection.isEmpty(), "Aliases of [%s] conflict with aliases of [%s]. Conflicting: %s", this, pd, intersection);
                }
            }
        }
    }

    private void verifyName(Verifications verifications, String name) {
        verifications.verify(!this.name.equals(name), "Cannot define a property named '%s' on %s", name, declaringDescriptor);
    }

    private static void initInspectionMetadata(Field field, LocalPropertyDescriptor propertyDescriptor) {
        if (field.isAnnotationPresent(InspectionProperty.class)) {
            propertyDescriptor.inspectionProperty = true;
            propertyDescriptor.requiredForInspection = field.getAnnotation(InspectionProperty.class).required();
        }
    }

    private void initMetadata(Property annotation) {
        category = annotation.category();
        label = isBlank(annotation.label()) ? deCamelize(name) : annotation.label();
        description = isBlank(annotation.description()) ? label : annotation.description();
        password = annotation.password();
        required = annotation.required();
        size = annotation.size();
        hidden = annotation.hidden();
        isTransient = hidden || annotation.isTransient();
    }

    private void initType(Property annotation) {
        Class<?> type = field.getType();
        if (type == boolean.class) {
            kind = BOOLEAN;
        } else if (type == int.class || type == Integer.class) {
            kind = INTEGER;
        } else if (type == String.class) {
            kind = STRING;
        } else if (type.isEnum()) {
            kind = ENUM;
            initEnumValues(field.getType());
        } else if (type == Date.class) {
            kind = DATE;
        } else if (ConfigurationItem.class.isAssignableFrom(type)) {
            kind = CI;
            referencedType = Type.valueOf(type);
            asContainment = annotation.asContainment();
        } else if (Set.class.isAssignableFrom(type)) {
            initSetType(annotation);
        } else if (Map.class.isAssignableFrom(type)) {
            initMapType(annotation);
        } else if (List.class.isAssignableFrom(type)) {
            initListType(annotation);
        } else {
            throw new IllegalArgumentException(format("Type of %s not supported as an @Property field, found on %s.%s", type.getName(), field
                    .getDeclaringClass().getName(), name));
        }

        GlobalContext.register(this, emptyToNull(annotation.defaultValue()));
    }

    private Object convertValue(String val) {
        if (val == null) return null;
        switch (kind) {
            case BOOLEAN:
                return Boolean.parseBoolean(val);
            case INTEGER:
                if (val.isEmpty()) return null;
                return Integer.parseInt(val);
            case STRING:
                return val;
            case ENUM:
                for (Enum<?> enumConstant : (Enum<?>[]) enumClass.getEnumConstants()) {
                    if (enumConstant.name().equalsIgnoreCase(val)) {
                        return enumConstant;
                    }
                }
                throw new IllegalArgumentException("Value " + val + " not a member of enum " + enumClass);
            case DATE:
                return DatatypeConverter.parseDateTime(val).getTime();
            case SET_OF_STRING:
                return newLinkedHashSet(splitValue(val));
            case LIST_OF_STRING:
                return newArrayList(splitValue(val));
            case MAP_STRING_STRING:
                return decodeMap(val);
            default:
                throw new IllegalArgumentException("Property " + name + " of kind " + kind + " cannot be converted from a string value");
        }
    }

    private static Iterable<String> splitValue(String val) {
        return Splitter.on(',').trimResults().omitEmptyStrings().split(val);
    }

    private Map<String, String> decodeMap(String val) {
        Map<String, String> map = newLinkedHashMap();
        // TODO maybe create smart parser with quote support...
        for (String s : splitValue(val)) {
            String[] split = s.split(":");
            checkArgument(emptyToNull(split[0]) != null, "Property %s of kind %s cannot be converted from string value '%s' because of any empty key.", name, kind, val);
            if (split.length == 1) {
                map.put(split[0], "");
            } else {
                map.put(split[0], split[1]);
            }
        }
        return map;
    }

    private void initSetType(Property annotation) {
        Class<?> setType = getGenericType(Set.class, 1, 0);
        if (setType == String.class) {
            kind = SET_OF_STRING;
        } else if (ConfigurationItem.class.isAssignableFrom(setType)) {
            kind = SET_OF_CI;
            referencedType = Type.valueOf(setType);
            asContainment = annotation.asContainment();
        } else {
            throw new IllegalStateException(format("Unsupported Set type encountered for [%s]. Only support String and ConfigurationItem", name));
        }
    }

    private void initMapType(Property annotation) {
        checkArgument(getGenericType(Map.class, 2, 0) == String.class, "Property %s.%s of type Map should be Map<String, String>", field.getDeclaringClass().getName(), name);
        checkArgument(getGenericType(Map.class, 2, 1) == String.class, "Property %s.%s of type Map should be Map<String, String>", field.getDeclaringClass().getName(), name);
        kind = MAP_STRING_STRING;
    }

    private void initListType(Property annotation) {
        Class<?> listType = getGenericType(List.class, 1, 0);
        if (listType == String.class) {
            kind = LIST_OF_STRING;
        } else if (ConfigurationItem.class.isAssignableFrom(listType)) {
            kind = LIST_OF_CI;
            referencedType = Type.valueOf(listType);
            asContainment = annotation.asContainment();
        } else {
            throw new IllegalStateException(format("Unsupported List type encountered for [%s]. Only support String and ConfigurationItem", name));
        }
    }

    private Class<?> getGenericType(Class<?> collectionClass, int nrExpectedTypes, int indexOfType) {
        java.lang.reflect.Type genericType = field.getGenericType();
        checkArgument(genericType instanceof ParameterizedType, "The field %s.%s is a %s but it isn't a generic type (%s)",
                field.getDeclaringClass().getName(), name, collectionClass, genericType);
        java.lang.reflect.Type[] actualTypeArguments = ((ParameterizedType) genericType).getActualTypeArguments();
        checkArgument(actualTypeArguments.length == nrExpectedTypes, "The field %s is a %s.%s but it doesn't have the right generic type (%s)", field.getDeclaringClass()
                .getName(), name, collectionClass, actualTypeArguments);
        checkArgument(actualTypeArguments[indexOfType] instanceof Class, "The field %s.%s is a %s but it is not a concrete subclass (%s)", field.getDeclaringClass()
                .getName(), name, collectionClass, Arrays.toString(actualTypeArguments));
        return (Class<?>) actualTypeArguments[indexOfType];

    }

    private void initEnumValues(Class<?> enumClass) {
        enumValues = newArrayList();
        this.enumClass = enumClass;
        for (Enum<?> enumValue : (Enum<?>[]) enumClass.getEnumConstants()) {
            enumValues.add(enumValue.name());
        }
    }

    void overrideWith(LocalPropertyDescriptor superDescriptor) {
        checkArgument(
                superDescriptor.getKind() == this.getKind(),
                "Type '%s' attempts to overrides property '%s' declared in type '%s', but kind attribute does not match. Derived kind: '%s'. Super kind: '%s'.",
                getDeclaringDescriptor().getType(), getName(),
                superDescriptor.getDeclaringDescriptor().getType(),
                getKind(), superDescriptor.getKind());
        checkOverrideArgument((superDescriptor.getReferencedType() == null && this.getReferencedType() == null) ||
                (superDescriptor.getReferencedType() != null && superDescriptor.getReferencedType().equals(this.getReferencedType())) ||
                (superDescriptor.getReferencedType() != null && superDescriptor.getReferencedType().isSuperTypeOf(this.getReferencedType())), "referenceType", superDescriptor.getReferencedType() + " or a subtype thereof", superDescriptor);
        checkOverrideArgument(superDescriptor.enumClass == this.enumClass, "enumClass", superDescriptor.enumClass, superDescriptor);


        // Copy these properties verbatim from the previous version of the descriptor
        field = superDescriptor.field;
        asContainment = superDescriptor.asContainment;
        password = superDescriptor.password;
        if (superDescriptor.isRequired()) {
            // Special case, we're marking this property as required, need to add the validation rule then.
            if (!required) {
                validationRules.add(new RequiredValidator());
            }
            required = true;
        }
        if (superDescriptor.inspectionProperty) {
            inspectionProperty = true;
            requiredForInspection = superDescriptor.requiredForInspection;
        }

        // In type modifications, the description and label might not be set, copy the description and/or the label from the previous version of the descriptor in that case.
        if (description.equals(label)) {
            description = superDescriptor.description;
        }
        if (label.equals(deCamelize(name))) {
            label = superDescriptor.label;
        }
        if (category.equals("Common")) {
            category = superDescriptor.category;
        }
        if (size == Property.Size.DEFAULT) {
            size = superDescriptor.size;
        }
    }

    private void checkOverrideArgument(boolean condition, String attribute, Object expectedValue, PropertyDescriptor superDescriptor) {
        final String attributeErrorTemplate = "Type '%s' attempts to overrides property '%s' declared in type '%s', but '%s' attribute does not match that in the super type. Should be set to %s.";
        checkArgument(condition, attributeErrorTemplate, getDeclaringDescriptor().getType(), getName(), ((LocalPropertyDescriptor) superDescriptor).getDeclaringDescriptor().getType(), attribute, expectedValue);
    }

    private void initSynthetic(Element propertyElement) {
        kind = PropertyKind.valueOf(getOptionalStringAttribute(propertyElement, "kind", STRING.name()).toUpperCase());
        category = getOptionalStringAttribute(propertyElement, "category", "Common");
        label = getOptionalStringAttribute(propertyElement, "label", deCamelize(name));
        description = getOptionalStringAttribute(propertyElement, "description", label);
        required = getOptionalBooleanAttribute(propertyElement, "required", true);
        password = getOptionalBooleanAttribute(propertyElement, "password", false);
        asContainment = getOptionalBooleanAttribute(propertyElement, "as-containment", false);
        size = Property.Size.valueOf(getOptionalStringAttribute(propertyElement, "size", Property.Size.DEFAULT.name()).toUpperCase());
        String defaultValueAttr = getOptionalStringAttribute(propertyElement, "default", null);
        hidden = getOptionalBooleanAttribute(propertyElement, "hidden", false);
        inspectionProperty = getOptionalBooleanAttribute(propertyElement, "inspectionProperty", false);
        aliases = newHashSet(Splitter.on(",").omitEmptyStrings().split(getOptionalStringAttribute(propertyElement, "aliases", "")));
        isTransient = hidden || getOptionalBooleanAttribute(propertyElement, "transient", false);

        if (kind == ENUM) {
            String enumClassAttributeValue = getRequiredStringAttribute(propertyElement, "enum-class", "for property " + name + " of kind " + kind);
            try {
                enumClass = Class.forName(enumClassAttributeValue);
                if (!enumClass.isEnum()) {
                    throw new IllegalArgumentException("enum-class supplied for property " + name + " of kind " + kind + " is not an enum: "
                            + enumClassAttributeValue);
                }
                initEnumValues(enumClass);
            } catch (ClassNotFoundException exc) {
                throw new IllegalArgumentException("Unknown enum-class supplied for property " + name + " of kind " + kind + ": " + enumClassAttributeValue,
                        exc);
            }
        }

        if (kind == CI || kind == SET_OF_CI || kind == LIST_OF_CI) {
            referencedType = Type.valueOf(getRequiredStringAttribute(propertyElement, "referenced-type", "for property " + name + " of kind " + kind));
        }

        initSyntheticValidationRules(propertyElement);
        addDefaultValidationRules();
        GlobalContext.register(this, defaultValueAttr);
    }

    private void addDefaultValidationRules() {
        if (required && !asContainment && kind != PropertyKind.BOOLEAN) {
            validationRules.add(new RequiredValidator());
        }

        if (of(SET_OF_CI, SET_OF_STRING, LIST_OF_CI, LIST_OF_STRING).contains(kind)) {
            validationRules.add(new CollectionTypeValidator(this));
        }
        
        if (of(SET_OF_CI, LIST_OF_CI).contains(kind)) {
            validationRules.add(new ReferenceCollectionTypeValidator(this));
        } else if (CI == kind) {
            validationRules.add(new ReferenceTypeValidator(this));
        }
    }

    private void initSyntheticValidationRules(Element propertyElement) {
        forEach(childByName(propertyElement, Predicates.equalTo("rule")), new Closure<Element>() {
            public void call(Element element) {
                validationRules.add(ValidationRuleConverter.makeRule(element, LocalPropertyDescriptor.this));
            }
        });
    }

    static PropertyDescriptor generateDeployableFrom(LocalDescriptor descriptor, PropertyDescriptor deployedPropertyDescriptor) {
        LocalPropertyDescriptor propertyDescriptor = new LocalPropertyDescriptor(deployedPropertyDescriptor.getName());
        propertyDescriptor.declaringDescriptor = descriptor;
        propertyDescriptor.generateDeployable(deployedPropertyDescriptor);
        return propertyDescriptor;
    }

    private void generateDeployable(PropertyDescriptor deployedPropertyDescriptor) {
        // Generated 'simple' Deployable properties always are of kind String so that placeholder/dictionaries values can be filled.
        kind = deployedPropertyDescriptor.getKind().isSimple() ? PropertyKind.STRING : deployedPropertyDescriptor.getKind();
        category = deployedPropertyDescriptor.getCategory();
        label = deployedPropertyDescriptor.getLabel();
        description = format("%s (%s)", deployedPropertyDescriptor.getDescription(), deployedPropertyDescriptor.getKind().name().toLowerCase());
        required = false;
        password = deployedPropertyDescriptor.isPassword();
        size = deployedPropertyDescriptor.getSize();
        field = null;
    }

    @Override
    public String getName() {
        return name;
    }

    @Override
    public String getDescription() {
        return description;
    }

    @Override
    public boolean isAsContainment() {
        return asContainment;
    }

    @Override
    public String getCategory() {
        return category;
    }

    @Override
    public String getLabel() {
        return label;
    }

    @Override
    public boolean isPassword() {
        return password;
    }

    @Override
    public boolean isRequired() {
        return required;
    }

    @Override
    public Property.Size getSize() {
        return size;
    }

    @Override
    public PropertyKind getKind() {
        return kind;
    }

    @Override
    public List<String> getEnumValues() {
        return enumValues;
    }

    public Descriptor getDeclaringDescriptor() {
        return declaringDescriptor;
    }

    @Override
    public Type getReferencedType() {
        return referencedType;
    }

    @Override
    public Object getDefaultValue() {
        return convertValue(GlobalContext.lookup(this));
    }

    @Override
    public Set<String> getAliases() {
        return aliases;
    }

    @Override
    public boolean isHidden() {
        return hidden;
    }

    @Override
    @SuppressWarnings("unchecked")
    public Object get(ConfigurationItem item) {
        if (field != null) {
            try {
                return field.get(item);
            } catch (IllegalAccessException e) {
                throw new RuntimeException("Cannot get field " + field, e);
            }
        } else {
            Map<String, Object> synth = (Map<String, Object>) ReflectionUtils.getField(item, declaringDescriptor.getSyntheticPropertiesField());
            return synth.get(name);
        }
    }

    @Override
    @SuppressWarnings("unchecked")
    public void set(ConfigurationItem item, Object value) {
        if (value instanceof String) {
            value = convertValue((String) value);
        } else if (value == null) {
            value = getDefaultValue();
        }

        logger.trace("Setting value [{}] on property [{}] of CI [{}]", value, getFqn(), item.getId());

        try {
            if (field != null) {
                if (value == null) {
                    return;
                }
                field.set(item, value);
            } else {
                Map<String, Object> synth = (Map<String, Object>) ReflectionUtils.getField(item, declaringDescriptor.getSyntheticPropertiesField());
                synth.put(name, value);
            }
        } catch (IllegalAccessException e) {
            throw new RuntimeException("Cannot set field " + field, e);
        }
    }

    @Override
    public boolean areEqual(ConfigurationItem item, ConfigurationItem other) {
        return areEqual(item, other, new HashSet<String>());
    }

    boolean areEqual(ConfigurationItem item, ConfigurationItem other, Set<String> itemsBeingCompared) {
        Object left = get(item);
        Object right = get(other);
        logger.trace("Comparing {}: old [{}] <-> new [{}]", getFqn(), left, right);
        return areValuesEqual(left, right, itemsBeingCompared);
    }

    @SuppressWarnings("unchecked")
    private boolean areValuesEqual(Object itemValue, Object otherValue, Set<String> itemsBeingCompared) {

        if (itemValue == null) {
            return otherValue == null;
        } else if (otherValue == null) {
            return false;
        }

        switch (kind) {
            case SET_OF_STRING:
                return symmetricDifference((Set<String>) itemValue, (Set<String>) otherValue).isEmpty();
            case SET_OF_CI:
                Function<ConfigurationItem, String> f = new Function<ConfigurationItem, String>() {
                    @Override
                    public String apply(ConfigurationItem from) {
                        return from.getName();
                    }
                };

                Iterable<ConfigurationItem> cis = Iterables.concat((Set<ConfigurationItem>) itemValue, (Set<ConfigurationItem>) otherValue);
                Multimap<String, ConfigurationItem> index = Multimaps.index(cis, f);
                for (String key : index.keySet()) {
                    Collection<ConfigurationItem> cisToCompare = index.get(key);
                    if (cisToCompare.size() != 2) {
                        return false;
                    }
                    Iterator<ConfigurationItem> itemIterator = cisToCompare.iterator();
                    ConfigurationItem lhs = itemIterator.next();
                    ConfigurationItem rhs = itemIterator.next();
                    if (areCiEqual(lhs, rhs, itemsBeingCompared)) continue;
                    return false;
                }
                return true;
            case CI:
                ConfigurationItem itemValueAsCi = (ConfigurationItem) itemValue;
                ConfigurationItem otherValueAsCi = (ConfigurationItem) otherValue;
                if (itemValueAsCi.getName().equals(otherValueAsCi.getName())) {
                    LocalDescriptor descriptor = (LocalDescriptor) LocalDescriptorRegistry.getDescriptor(itemValueAsCi.getType());
                    return descriptor.areEqual(itemValueAsCi, otherValueAsCi, itemsBeingCompared);
                }
                return false;
            case MAP_STRING_STRING:
                Map<String, String> left = (Map<String, String>) itemValue;
                Map<String, String> right = (Map<String, String>) otherValue;
                return Maps.difference(left, right).areEqual();
            case LIST_OF_STRING:
                List<String> leftStrings = (List<String>) itemValue;
                List<String> rightStrings = (List<String>) otherValue;
                return leftStrings.equals(rightStrings);
            case LIST_OF_CI:
                List<ConfigurationItem> lhs = (List<ConfigurationItem>) itemValue;
                List<ConfigurationItem> rhs = (List<ConfigurationItem>) otherValue;
                if (lhs.size() != rhs.size()) {
                    return false;
                } else {
                    Iterator<ConfigurationItem> lIter = lhs.iterator();
                    Iterator<ConfigurationItem> rIter = rhs.iterator();
                    ConfigurationItem lItem;
                    ConfigurationItem rItem;
                    while (lIter.hasNext()) {
                        lItem = lIter.next();
                        rItem = rIter.next();
                        if (areCiEqual(lItem, rItem, itemsBeingCompared)) continue;
                        return false;
                    }
                    return true;
                }
            default:
                return itemValue.equals(otherValue);
        }
    }

    private boolean areCiEqual(ConfigurationItem lItem, ConfigurationItem rItem, Set<String> itemsBeingCompared) {
        if (lItem.getName().equals(rItem.getName()) && lItem.getType().equals(rItem.getType())) {
            LocalDescriptor descriptor = (LocalDescriptor) LocalDescriptorRegistry.getDescriptor(lItem.getType());
            if (descriptor.areEqual(lItem, rItem, itemsBeingCompared)) {
                return true;
            }
        }
        return false;
    }

    @SuppressWarnings("unchecked")
    void validate(final ConfigurationItem ci, final List<ValidationMessage> messages) {
        ValidationContext context = new ValidationContext() {
            @Override
            public void error(String message, Object... params) {
                messages.add(new ValidationMessage(ci.getId(), name, format(message, params)));
            }
        };
        for (Validator<?> validationRule : validationRules) {
            ((Validator<Object>) validationRule).validate(get(ci), context);
        }
    }

    @Override
    public boolean isInspectionProperty() {
        return inspectionProperty;
    }

    @Override
    public boolean isRequiredForInspection() {
        return requiredForInspection;
    }

    @Override
    public boolean isTransient() {
        return isTransient;
    }

    @Override
    public String getFqn() {
        return declaringDescriptor.getType() + "." + name;
    }

    @Override
    public String toString() {
        return getFqn();
    }

    private static final Logger logger = LoggerFactory.getLogger(LocalPropertyDescriptor.class);

    public Object emptyValue() {
        switch (kind) {
            case BOOLEAN:
                return false;
            case SET_OF_STRING:
            case SET_OF_CI:
                return newHashSet();
            case LIST_OF_STRING:
            case LIST_OF_CI:
                return newArrayList();
            case MAP_STRING_STRING:
                return newHashMap();
            default:
                return null;
        }
    }

    public Field getField() {
        return field;
    }

}
