/**
 * Copyright 2014-2019 XebiaLabs Inc. and its affiliates. Use is subject to terms of the enclosed Legal Notice.
 */
package com.xebialabs.deployit.packager;

import java.io.IOException;
import java.io.InputStream;
import java.io.OutputStream;
import java.security.MessageDigest;
import java.util.Collection;

import com.xebialabs.xldeploy.packager.io.ArtifactIOUtils;
import com.xebialabs.xldeploy.packager.placeholders.SourceArtifactScanner;

import org.apache.commons.io.IOUtils;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;
import com.typesafe.config.ConfigFactory;

import com.xebialabs.deployit.plugin.api.reflect.PropertyDescriptor;
import com.xebialabs.deployit.plugin.api.udm.ConfigurationItem;
import com.xebialabs.deployit.plugin.api.udm.DeploymentPackage;
import com.xebialabs.deployit.plugin.api.udm.Version;
import com.xebialabs.deployit.plugin.api.udm.artifact.Artifact;
import com.xebialabs.deployit.plugin.api.udm.artifact.FolderArtifact;
import com.xebialabs.deployit.plugin.api.udm.artifact.SourceArtifact;
import com.xebialabs.overthere.OverthereFile;
import com.xebialabs.overthere.RuntimeIOException;
import com.xebialabs.xldeploy.packager.io.StreamerFactory;

import de.schlichtherle.truezip.file.TArchiveDetector;
import de.schlichtherle.truezip.file.TFile;
import de.schlichtherle.truezip.file.TFileOutputStream;
import de.schlichtherle.truezip.file.TVFS;
import de.schlichtherle.truezip.fs.FsSyncException;
import scala.Function0;

import static com.xebialabs.deployit.plugin.api.reflect.PropertyKind.LIST_OF_CI;
import static com.xebialabs.deployit.plugin.api.reflect.PropertyKind.SET_OF_CI;
import static java.lang.String.format;
import static java.util.EnumSet.of;

public class DarPackager {

    public static final String PRESCANNED_PLACEHOLDERS_PROPERTYNAME = "preScannedPlaceholders";

    private static final Logger logger = LoggerFactory.getLogger(DarPackager.class);
    private final CiNameCache cache = new CiNameCache();
    private ManifestWriterDriver manifestDriver;
    private Function0<MessageDigest> messageDigest;
    private SourceArtifactScanner scanner;
    private final byte[] byteBuffer = new byte[IOUtils.DEFAULT_BUFFER_SIZE];

    public DarPackager(ManifestWriter writer, Function0<MessageDigest> messageDigest) {
        this.manifestDriver = new ManifestWriterDriver(cache, writer);
        this.messageDigest = messageDigest;
        this.scanner = new SourceArtifactScanner(StreamerFactory.forConfig(ConfigFactory.defaultReference()));
    }

    public TFile buildPackage(Version version, String targetDir) {
        return buildPackage(version, targetDir, false);
    }

    public TFile buildPackage(Version version, String targetDir, boolean writeFullPaths) {
        if (version instanceof DeploymentPackage) {
            enrichArtifactsForDeploymentPackage((DeploymentPackage) version);
        }
        String darName = darName(version);
        TFile darPackage = new TFile(targetDir, darName);
        try {
            manifestDriver.writeToPackage(version, darPackage, writeFullPaths);
            writeArtifactsToPackage(version, darPackage);
        } catch (IOException e) {
            throw new RuntimeIOException(e);
        } finally {
            try {
                TVFS.umount(darPackage);
            } catch (FsSyncException e) {
                logger.warn("Couldn't cleanly umount the dar package {}", darPackage);
            }
        }
        return darPackage;
    }

    @SuppressWarnings("unchecked")
    private void writeArtifactsToPackage(ConfigurationItem ci, TFile darPackage) {
        if (manifestDriver.isLocalArtifact(ci)) {
            if (ci instanceof FolderArtifact) {
                OverthereFile sourceFile = ((Artifact) ci).getFile();
                if (logger.isDebugEnabled()) {
                    logger.debug("Writing {} to package {}", sourceFile, darPackage);
                }
                TFile artifactDir = new TFile(darPackage, cache.lookup(ci), TArchiveDetector.NULL);
                artifactDir.mkdirs();
                if (sourceFile.isFile()) {
                    copyFile(sourceFile, new TFile(artifactDir, sourceFile.getName(), TArchiveDetector.NULL));
                } else {
                    TFile targetFolder = new TFile(artifactDir, sourceFile.getName());
                    targetFolder.mkdirs();
                    copyFolder(sourceFile, targetFolder);
                }
            } else if (ci instanceof Artifact) {
                OverthereFile sourceFile = ((Artifact) ci).getFile();
                if (logger.isDebugEnabled()) {
                    logger.debug("Writing {} to package {}", sourceFile, darPackage);
                }
                TFile artifactDir = new TFile(darPackage, cache.lookup(ci), TArchiveDetector.NULL);
                artifactDir.mkdirs();
                TFile targetFile = new TFile(artifactDir, sourceFile.getName(), TArchiveDetector.NULL);
                copyFile(sourceFile, targetFile);
            }
        }
        for (PropertyDescriptor pd : ci.getType().getDescriptor().getPropertyDescriptors()) {
            if (pd.isAsContainment() && of(LIST_OF_CI, SET_OF_CI).contains(pd.getKind())) {
                Collection<ConfigurationItem> cis = (Collection<ConfigurationItem>) pd.get(ci);
                if (cis == null) {
                    logger.warn("{}: {}.{} (as-containment, kind {}) should not be null",
                            ci.getId(), ci.getType(), pd.getName(), pd.getKind());
                    continue;
                }
                for (ConfigurationItem c : cis) {
                    writeArtifactsToPackage(c, darPackage);
                }
            }
        }
    }

    private void enrichArtifactsForDeploymentPackage(DeploymentPackage version) {
        version.getDeployables().stream().filter(d -> d instanceof SourceArtifact).forEach(deployable -> {
            SourceArtifact sourceArtifact = (SourceArtifact) deployable;
            if (artifactContainsRealFile(sourceArtifact) && !isArtifactPreEnriched(sourceArtifact)) {
                if (logger.isInfoEnabled()) {
                    logger.info("Going to enrich artifact: {}", sourceArtifact);
                }
                scanner.enrichArtifact(sourceArtifact, scala.Option.empty(), messageDigest);
            }

        });
    }

    private Boolean isArtifactPreEnriched(SourceArtifact deployable) {
        Boolean isPreEnrichedArtifact =
                deployable.getChecksum() != null
                        && !deployable.getChecksum().isEmpty()
                        && deployable.hasProperty(DarPackager.PRESCANNED_PLACEHOLDERS_PROPERTYNAME)
                        && ((Boolean) deployable.getProperty(DarPackager.PRESCANNED_PLACEHOLDERS_PROPERTYNAME));
        if (logger.isInfoEnabled()) {
            logger.info("Artifact {} is {} pre-enriched", deployable, isPreEnrichedArtifact ? "" : "not");
        }
        return isPreEnrichedArtifact;
    }

    private boolean artifactContainsRealFile(SourceArtifact sourceArtifact) {
        boolean isFileReal = sourceArtifact.getFile() != null;
        if (logger.isInfoEnabled()) {
            logger.info("Artifact: {} is {} containing real file", sourceArtifact, isFileReal ? "" : "not");
        }
        return isFileReal;
    }


    private void copyFolder(OverthereFile sourceFile, TFile targetFolder) {
        for (OverthereFile overthereFile : sourceFile.listFiles()) {
            if (overthereFile.isFile()) {
                copyFile(overthereFile, new TFile(targetFolder, overthereFile.getName(), TArchiveDetector.NULL));
            } else if (overthereFile.isDirectory()) {
                TFile targetFolder1 = new TFile(targetFolder, overthereFile.getName());
                targetFolder1.mkdirs();
                copyFolder(overthereFile, targetFolder1);
            }
        }
    }

    private void copyFile(final OverthereFile sourceFile, TFile targetFile) {
        try (InputStream is = sourceFile.getInputStream(); OutputStream os = new TFileOutputStream(targetFile)) {
            ArtifactIOUtils.copyBytes(is, os, byteBuffer);
        } catch (IOException e) {
            throw new RuntimeIOException(e);
        }
    }

    private String darName(Version version) {
        return format("%s-%s.dar", version.getApplication().getName(), version.getVersion());
    }
}
