/**
 * Copyright 2014-2019 XebiaLabs Inc. and its affiliates. Use is subject to terms of the enclosed Legal Notice.
 */
package com.xebialabs.xldeploy.packager

import java.io.{IOException, Reader}

import com.xebialabs.xldeploy.packager.Delims._
import com.xebialabs.xldeploy.packager.MustacherReplacer._
import grizzled.slf4j.Logging

import scala.annotation.tailrec

class MustacheReplacingReader(mustacher: MustacherReplacer, wrapped: Reader, placeholders: Map[String, String]) extends Reader {
  private val endOfStreamValue: Int = -1

  override def read(chars: Array[Char]): Int = read(chars, 0, chars.length)

  override def read(chars: Array[Char], off: Int, len: Int): Int = {
    var charsRead: Int = 0
    var i: Int = 0
    while ( {
      i < len
    }) {
      val nextChar: Int = read()
      if (nextChar == endOfStreamValue) {
        if (charsRead == 0) {
          charsRead = endOfStreamValue
        }
        return charsRead
      }
      charsRead = i + 1
      chars(off + i) = nextChar.toChar

      i += 1
    }
    charsRead
  }

  @tailrec
  override final def read(): Int = {
    mustacher.handleReplace(wrapped, placeholders) match {
      case Letter(value) => value
      case EndOfStream => endOfStreamValue
      case Nothing => read()
    }
  }

  override def close(): Unit = {
    wrapped.close()
    mustacher.resetState()
  }
}

object MustacherReplacer extends MustacheReader {
  def apply(delims: String): MustacherReplacer = new MustacherReplacer(new Delims(delims))
  sealed trait BufferState
  final case class Letter(c: Int) extends BufferState
  case object Nothing extends BufferState
  case object EndOfStream extends BufferState
}

class MustacherReplacer(delims: Delims) extends Logging {
  var placeholders: Set[String] = Set.empty
  var state: MustacherReplacer.State = MustacherReplacer.Text
  private var lookBehind: Char = _
  private val invalids = Set('\r', '\n')

  var replacementBuffer: Option[String] = None
  var replacementBufferPosition = 0

  def newReader(reader: Reader, placeholders: Map[String, String]): MustacheReplacingReader = new MustacheReplacingReader(this, reader, placeholders)

  def handleReplace(reader: Reader, replacements: Map[String, String]): BufferState = {
    if (replacementBuffer.isDefined) {
      readFromReplacementBuffer()
    } else {
      readFromOriginalBuffer(reader, replacements)
    }
  }

  def resetState(): Unit = {
    state = Text
  }

  def resolve(tag: String, placeholders: Map[String, String]): String = {
      placeholders.get(tag) match {
        case Some(rb: String) if rb == IGNORE_PLACEHOLDER => delims.start + tag + delims.end
        case Some(rb: String) if rb == EMPTY_PLACEHOLDER => ""
        case Some(rb: String) => rb
        case _ => throw new IOException(s"Could not find a replacement for [$tag] placeholder.")
      }
  }

  private def readFromReplacementBuffer(): BufferState = {
    replacementBuffer match {
      case Some(replacement) if replacement.length > 0 && replacementBufferPosition < replacement.length =>
        val result = replacement.charAt(replacementBufferPosition)
        replacementBufferPosition += 1
        Letter(result)
      case _ =>
        replacementBuffer = None
        replacementBufferPosition = 0
        Nothing
    }
  }

  private def readFromOriginalBuffer(reader: Reader, replacements: Map[String, String]): BufferState  = {
    val data: Int = reader.read()
    if (data != -1) {
      val c = data.toChar
      val result = state match {
        case Text if delims.matchesStart(c) == Partial => // 2-char start delimiter
          state = MatchStartTag
          Nothing
        case Text if delims.matchesStart(c) == Full => // 1-char start delimiter
          state = MatchingTag(Array())
          Nothing
        case Text => // Ok
          Letter(c.toInt)
        case MatchStartTag if delims.matchesStart(lookBehind, c) == Full => // 2-char start delimiter
          state = MatchingTag(Array())
          Nothing
        case MatchStartTag => // Not matched delimiter, continue in text mode
          replacementBuffer = Option(Seq(delims.start(0), c).mkString)
          state = Text
          Nothing
        case MatchingTag(tag) if delims.matchesEnd(c) == Full => // 1-char end delimiter
          val stringTag = tag.mkString.trim
          logger.debug(s"Found placeholder: [$stringTag]")
          placeholders += stringTag
          val replaced = resolve(stringTag, replacements)
          replacementBuffer = Option(replaced)
          logger.debug(s"It should be replaced with: [$replaced]")
          state = Text
          Nothing
        case MatchingTag(cs) if delims.matchesEnd(c) == Partial => // 2-char end delimiter
          state = MatchEndTag(cs)
          Nothing
        case MatchingTag(cs) if delims.matchesStart(c) == Partial => // 2-char nested delimiter start with partial match for previously found
          state = MatchNestedStart(cs)
          Nothing
        case MatchingTag(cs) if delims.matchesStart(c) == Full => // 2-char nested delimiter start with full match for previously found and possibly some characters scanned
          replacementBuffer = Option(delims.start + cs.mkString)
          state = MatchingTag(Array()) // Throw away currently matched tag
          Nothing
        case MatchingTag(cs) if invalids.contains(c) =>
          // No support for multiline placeholders, stop processing
          logger.debug(s"Found and skipping invalid placeholder with linefeed and/or carriage return character: [${cs.mkString.trim}]")
          replacementBuffer = Option(delims.start + cs.mkString + c)
          state = Text
          Nothing
        case MatchingTag(cs) =>
          state = MatchingTag(cs :+ c)
          Nothing
        case MatchNestedStart(cs) if delims.matchesStart(lookBehind, c) == Full => // 2-char nested delimiter start
          val prefix = if (cs.isEmpty) delims.start(0) else delims.start
          replacementBuffer = Option(prefix + cs.mkString)
          state = MatchingTag(Array()) // Throw away currently matched tag
          Nothing
        case MatchNestedStart(cs) if cs.isEmpty => // Not matched full nested start-delimiter with empty placeholder tag, continue tag-matching
          replacementBuffer = Option(delims.start(0).toString)
          state = MatchingTag(Array[Char](1) :+ c)
          Nothing
        case MatchNestedStart(cs) if !cs.isEmpty => // Not matched full nested start-delimiter, continue tag-matching
          val chars = cs :+ delims.start(0)
          state = MatchingTag(chars :+ c)
          Nothing
        case MatchEndTag(tag) if delims.matchesEnd(lookBehind, c) == Full => // 2-char end delimiter
          val stringTag = tag.mkString.trim
          logger.debug(s"Found placeholder: [$stringTag]")
          placeholders += stringTag
          val replaced = resolve(stringTag, replacements)
          replacementBuffer = Option(replaced)
          logger.debug(s"It should be replaced with: [${resolve(stringTag, replacements)}]")
          state = Text
          Nothing
        case MatchEndTag(tag) => // Not matched full end-delimiter, continue tag-matching
          state = MatchingTag(tag :+ lookBehind :+ c)
          Nothing
      }
      lookBehind = c
      result
    } else state match {
      case tag: MatchingTag =>
        replacementBuffer = Option(delims.start + tag.c.mkString)
        state = Text
        Nothing
      case tag: MatchNestedStart =>
        replacementBuffer = Option(delims.start + tag.c.mkString + delims.start(0))
        state = Text
        Nothing
      case tag: MatchEndTag =>
        replacementBuffer = Option(delims.start + tag.tag.mkString + delims.end(0))
        state = Text
        Nothing
      case _ =>
        state = Text
        EndOfStream
    }
  }
}