/**
 * Copyright 2014-2019 XebiaLabs Inc. and its affiliates. Use is subject to terms of the enclosed Legal Notice.
 */
package com.xebialabs.xldeploy.packager.transcode

import com.xebialabs.deployit.plugin.api.udm.artifact.{SourceArtifact, TranscodableSourceArtifact}
import com.xebialabs.overthere.ssh.SshZosConnectionType
import com.xebialabs.overthere.ConnectionOptions
import com.xebialabs.xldeploy.packager.io.{ArtifactIOUtils, StreamEntry}
import com.xebialabs.xldeploy.packager.transcode.TranscodeSpec.{EBCDIC_CHARSETS, compilePattern, compilePatternForDetectCharset, logger, patternMapForDetectCharsetToTranscode}
import grizzled.slf4j.Logger
import org.slf4j.LoggerFactory

import java.io.InputStream
import java.nio.charset.Charset
import java.util.concurrent.ConcurrentHashMap
import java.util.regex.Pattern
import java.util.regex.Pattern.{CASE_INSENSITIVE, COMMENTS}
import scala.jdk.CollectionConverters._

object TranscodeSpec {
  private val EBCDIC_CHARSETS = List("CP1047", "IBM1047")

  private[TranscodeSpec] val logger = new Logger(LoggerFactory.getLogger(classOf[TranscodeSpec]))

  private lazy val patternMapForDetectCharsetToTranscode = new ConcurrentHashMap[String, Pattern].asScala
  private lazy val patternMapForTranscode = new ConcurrentHashMap[String, Pattern].asScala


  private def compilePatternForDetectCharset(fileEncodings: Map[String, String]): Unit = {
    fileEncodings.foreach { case (regex, _) =>
      compilePattern(regex,patternMapForDetectCharsetToTranscode);
    }
  }

  private def compilePattern(regex: String): Pattern = {
    compilePattern(regex,patternMapForTranscode)
  }

  private def compilePattern(regex: String,patternMap: scala.collection.concurrent.Map[String, Pattern]): Pattern = {
    if (regex != null && regex.nonEmpty) {
      patternMap.get(regex) match {
        case Some(compiled) => compiled
        case None =>
          val compiled = Pattern.compile(regex, COMMENTS | CASE_INSENSITIVE)
          patternMap += (regex -> compiled)
          compiled
      }
    } else {
      null
    }
  }

}

final case class TranscodeSpec(destConnectionOptions: ConnectionOptions, sourceArtifactTranscode: TranscodableSourceArtifact, regexToTranscodeEbcdic:Pattern) {

  private val sourceArtifact = sourceArtifactTranscode.asInstanceOf[SourceArtifact]
  private val excludeFilesToTranscodeEbcdic = sourceArtifactTranscode.getExcludeFileNamesRegexToTranscode
  private val fileEncodings = sourceArtifact.getFileEncodings.asScala.toMap

  def this(destConnectionOptions: ConnectionOptions, sourceArtifact: TranscodableSourceArtifact) = {
    this(destConnectionOptions, sourceArtifact, compilePattern(sourceArtifact.getTextFileNamesRegexToTranscode))
    compilePatternForDetectCharset(sourceArtifactTranscode.asInstanceOf[SourceArtifact].getFileEncodings.asScala.toMap)
  }

  def isTranscodeEbcdicRequired(entry: StreamEntry): Boolean = {
    !isTranscodeExcluded(entry) && isDestinationHostRequireTranscodeToEbcdic && isTranscodeFilesToEbcdicFound(entry)
  }

  def isDestinationHostRequireTranscodeToEbcdic:Boolean = isZosConnection

    def detectTranscodeMode(entry: StreamEntry):TranscodeMode = {
    if (isTranscodeEbcdicRequired(entry)) {
      Ebcdic
    } else {
      Skip
    }
  }

 private def isZosConnection = {
    val zosConnectionType: SshZosConnectionType = destConnectionOptions.getOptional("zosConnectionType")
    zosConnectionType != null && zosConnectionType == SshZosConnectionType.SFTP
  }

  private def isTranscodeFilesToEbcdicFound(entry: StreamEntry): Boolean = {
    isFileForTranscodeEbcdic(entry) && !isFileEncodingMappedWithEbcdic(entry)
  }

  private def isFileForTranscodeEbcdic(entry: StreamEntry) = {
    regexToTranscodeEbcdic != null  && regexToTranscodeEbcdic.matcher(entry.getName).matches()
  }

  private def isFileEncodingMappedWithEbcdic(entry: StreamEntry): Boolean = {
    val fileEncodingsFoundInEbcdic =
      fileEncodings.filter { case (regex, encode) =>
        patternMapForDetectCharsetToTranscode.get(regex) match {
          case Some(compiled) => compiled.matcher(entry.getName).matches() && EBCDIC_CHARSETS.contains(encode)
          case None =>
            logger.debug(s"transcode:Pattern NOT found for $regex")
            entry.getName.matches(regex) && EBCDIC_CHARSETS.contains(encode)
        }
      }
      fileEncodingsFoundInEbcdic.nonEmpty
  }

  private def isTranscodeExcluded(entry: StreamEntry) = {
    excludeFilesToTranscodeEbcdic != null && entry.getName.matches(excludeFilesToTranscodeEbcdic)
  }


  def detectCharset(is: InputStream, entry: StreamEntry): Option[Charset] = {
    ArtifactIOUtils.detectCharset(sourceArtifact,is,entry)
  }
}
