/**
 * Copyright 2014-2019 XebiaLabs Inc. and its affiliates. Use is subject to terms of the enclosed Legal Notice.
 */
package com.xebialabs.xldeploy.packager


import com.typesafe.config.{Config, ConfigUtil}
import com.xebialabs.xldeploy.packager.PackagerConfig.{CONFIG_PATH, MAX_THRESHOLD_BYTES, configPrefix, logger}
import org.slf4j.LoggerFactory

import scala.jdk.CollectionConverters._

object PackagerConfig {
  private val configPrefix = "xl.artifact.placeholders"
  private val logger = LoggerFactory.getLogger(getClass)
  // Threshold constants
  private[packager] val MAX_THRESHOLD_BYTES = 2147483648L // 2GB
  private val CONFIG_PATH = "xl.artifact.inMemoryThreshold"
}

class PackagerConfig(config: Config) {

  private val packagerConfig = config.getConfig(configPrefix)

  val archiveExtensionMappings: Map[String, String] =
    packagerConfig.getConfig("archive-extensions")
      .entrySet()
      .asScala
      .foldLeft(Map[String, String]()) {
        case (m, e) =>
          val key = ConfigUtil.splitPath(e.getKey).get(0)
          val value = e.getValue.unwrapped().toString
          if (key == "tar" && value != "tar") {
            m + (key -> "tar")
          } else {
            m + (key -> value)
          }
      }

  val zipEncoding = packagerConfig.getString("zip-encoding")
  val jarEncoding = packagerConfig.getString("jar-encoding")

  lazy val inMemoryThreshold: Long = initializeInMemoryThresholdFromConfig

  private def initializeInMemoryThresholdFromConfig = {
    try {
      if (config != null && config.hasPath(CONFIG_PATH)) {
        val configuredThreshold = config.getLong(CONFIG_PATH)
        logger.debug(s"stream:2G:Found inMemoryThreshold in config: $configuredThreshold bytes")
        // Cap at 2GB max
        capThresholdAt2GB(configuredThreshold)

      } else {
        logger.debug(s"stream:2G:$CONFIG_PATH not found in config, using default 2GB")
        MAX_THRESHOLD_BYTES // Default 2GB
      }
    } catch {
      case ex: Exception =>
        logger.warn(s"stream:2G:Failed to retrieve inMemoryThreshold from config: ${ex.getMessage}", ex)
        MAX_THRESHOLD_BYTES // Default 2GB on error
    }
  }

  private def capThresholdAt2GB(threshold: Long): Long = {
    if (threshold > MAX_THRESHOLD_BYTES) {
      logger.warn(s"stream:2G:Configured threshold ($threshold) exceeds 2GB limit, capped to 2GB")
      MAX_THRESHOLD_BYTES
    } else {
      threshold
    }
  }
}
