package com.xebialabs.deployit.engine.tasker

import akka.actor._
import com.xebialabs.deployit.engine.api.execution.StepExecutionState
import com.xebialabs.deployit.engine.tasker.BlockExecutingActor.BlockStateChanged
import com.xebialabs.deployit.engine.tasker.messages.Stop
import akka.actor.Status.Failure
import com.xebialabs.deployit.engine.tasker.StepExecutingActor.messages.ExecuteStep
import com.xebialabs.deployit.engine.tasker.BlockExecutingActor.BlockDone
import com.xebialabs.deployit.engine.tasker.messages.Start
import com.xebialabs.deployit.engine.tasker.messages.Abort

object StepBlockExecutingActor {
  def props(task: Task, block: StepContainingBlock[_ <: Seq[TaskStep]], ctx: NestedTaskExecutionContext) = Props(classOf[StepBlockExecutingActor], task, block, ctx)
}

class StepBlockExecutingActor(task: Task, sb: StepContainingBlock[_ <: Seq[TaskStep]], ctx: NestedTaskExecutionContext) extends BaseExecutionActor {

  import context._
  import BlockState._

  var block: StepContainingBlock[_ <: Seq[TaskStep]] = sb
  val taskId = task.getId

//  val stepActor = createChild(StepExecutingActor.props(task, ctx), "step")

  def receive: Actor.Receive = {
    case Start(`taskId`) if Set(PENDING, STOPPED, ABORTED, FAILED).contains(block.state) =>
      updateStateAndNotify(BlockState.EXECUTING)
      executeStep(0, createChild(StepExecutingActor.props(task, ctx), "step"))
    case Start(`taskId`) if block.state == EXECUTED =>
      log.debug(s"Not Executing block [${block.id}] again as it is already [EXECUTED]")
      notifyAndDie(block.state, None)
    case Start(`taskId`) => sender ! Failure(new IllegalStateException(s"Can not execute a step block in state [${block.state}]"))
    case m @ _ =>
      log.error(s"Don't know what to do with [$m]")
      sender ! Failure(new IllegalStateException(s"$m"))
  }

  def executeStep(stepNr: Int, stepActor: ActorRef) {
    if (block.steps.size <= stepNr) {
      notifyAndDie(BlockState.EXECUTED, Option(stepActor))
    } else {
      val head = block.steps(stepNr)
      become(waitForStep(head, stepNr + 1, stepActor))
      stepActor ! ExecuteStep(head, ctx.stepContext(head))
    }
  }

  def waitForStep(step: TaskStep, nextStepNr: Int, stepActor: ActorRef): Actor.Receive = {
    case StepExecutionState.DONE | StepExecutionState.SKIPPED => executeStep(nextStepNr, stepActor)
    case StepExecutionState.PAUSED => notifyAndDie(BlockState.STOPPED, Option(stepActor))
    case StepExecutionState.FAILED => notifyAndDie(BlockState.FAILED, Option(stepActor))
    case Stop(`taskId`) =>
      log.info(s"Received [Stop] message")
      updateStateAndNotify(BlockState.STOPPING)
      become(stopAfterCurrentStep(step, stepActor))
    case Abort(`taskId`) =>
      doAbort(stepActor, step)
  }


  def doAbort(stepActor: ActorRef, step: TaskStep) {
    log.info(s"Received [Abort] message")
    updateStateAndNotify(BlockState.ABORTING)
    context.watch(stepActor)
    become(aborting(step))
    step.interruptRunner()
    stepActor ! Kill
  }

  def aborting(step: TaskStep): Actor.Receive = {
    case m @ Terminated(stepActor) =>
      log.info(s"Received [$m], going to aborted state")
      step.setState(StepExecutionState.FAILED)
      notifyAndDie(BlockState.ABORTED, None)
    case m @ _ =>
      log.info(s"Received [$m], ignoring")
  }

  def notifyAndDie(state: BlockState, stepActorOption: Option[ActorRef]) {
    if (state == EXECUTED) {
      self ! PoisonPill
    } else {
      become(receive)
    }

    stepActorOption.foreach(system.stop)
    updateStateAndNotify(state)
    parent ! BlockDone(task.getId, block)
  }

  def stopAfterCurrentStep(step: TaskStep, stepActor: ActorRef): Actor.Receive = {
    case Abort(`taskId`) =>
      doAbort(stepActor, step)
    case m @ _ => notifyAndDie(BlockState.STOPPED, Option(stepActor))
  }

  def updateStateAndNotify(newState: BlockState) {
    val oldState: BlockState = block.state
    if (oldState != newState) {
      block = block.newState(newState)
      val msg: BlockStateChanged = BlockStateChanged(task.getId, block, oldState, newState)
      log.debug(s"Sending BlockStateChange($oldState->$newState) message for [${block.id}]")
      parent ! msg
    }
  }

}
