package com.xebialabs.deployit.engine.tasker

import akka.actor._
import com.xebialabs.deployit.engine.api.execution.TaskExecutionState2._
import com.xebialabs.deployit.engine.api.execution.TaskExecutionState2
import org.joda.time.DateTime
import com.github.nscala_time.time.Imports._
import scala.concurrent.duration._
import com.xebialabs.deployit.engine.tasker.messages._
import com.xebialabs.deployit.engine.tasker.TaskManagingActor.messages.Recovered
import akka.actor.Status.Failure
import scala.Some
import com.xebialabs.deployit.engine.tasker.ArchiveActor.messages.SendToArchive
import com.xebialabs.deployit.engine.tasker.StateChangeEventListenerActor.StepStateEvent
import com.xebialabs.deployit.engine.tasker.messages.Cancelled
import com.xebialabs.deployit.engine.tasker.messages.Registered
import com.xebialabs.deployit.engine.tasker.TaskManagingActor.messages.Schedule
import com.xebialabs.deployit.engine.tasker.TaskManagingActor.messages.ArchiveTask
import com.xebialabs.deployit.engine.tasker.BlockExecutingActor.BlockStateChanged
import com.xebialabs.deployit.engine.tasker.TaskManagingActor.messages.ScheduledStart
import com.xebialabs.deployit.engine.tasker.messages.Stop
import com.xebialabs.deployit.engine.tasker.messages.TaskDone
import com.xebialabs.deployit.engine.tasker.TaskManagingActor.messages.Register
import com.xebialabs.deployit.engine.tasker.messages.Archived
import com.xebialabs.deployit.engine.tasker.TaskManagingActor.messages.Cancel
import com.xebialabs.deployit.engine.tasker.messages.TaskStateEventHandled
import com.xebialabs.deployit.engine.tasker.BlockExecutingActor.BlockDone
import com.xebialabs.deployit.engine.tasker.messages.Start
import com.xebialabs.deployit.engine.tasker.TaskManagingActor.messages.GetTask
import com.xebialabs.deployit.engine.tasker.StateChangeEventListenerActor.TaskStateEvent

object TaskManagingActor {
  def props = Props(classOf[TaskManagingActor])

  def getMessages = messages

  object messages {
    case class GetTask(taskId: TaskId)
    case class Register(task: Task)
    case class ScheduledStart(taskId: TaskId)
    case class Schedule(taskId: TaskId, scheduleAt: DateTime)
    case class ArchiveTask(taskId: TaskId, archiveActor: ActorRef, notificationActor: ActorRef)
    case class Cancel(taskId: TaskId, archiveActor: ActorRef, notificationActor: ActorRef)
    case class Recovered(task: Task)
  }
}

class TaskManagingActor extends BaseExecutionActor {
  import context._

  val terminationStates = Set(FAILED, ABORTED, EXECUTED, STOPPED)

  def updateStateAndNotify(t: Task, newState: TaskExecutionState2): Task = {
    log.debug(s"Sending TaskStateEvent(${t.state}->$newState) message for [${t.getId}]")
    implicit val system = context.system
    t.setTaskStateAndNotify(newState)
    t
  }

  def receive: Actor.Receive = {
    case Register(task) =>
      log.info(s"Received [Register] message for task [${task.getId}]")
      registerStateListeners(task)
      become(pending(task.getId, updateStateAndNotify(task, PENDING)))
      TaskRegistryExtension(system).register(task)
      sender ! Registered(task.getId)
    case Recovered(task) =>
      log.info(s"Received [Recovered] message for task [${task.getId}]")
      registerStateListeners(task)
      task.state match {
        case PENDING | QUEUED => become(pending(task.getId, task))
        case SCHEDULED if task.getScheduledDate.isAfterNow => doSchedule(task.getId, task, task.getScheduledDate, createOrLookupChildForTaskBlock(task))
        case SCHEDULED => doEnqueue(task.getId, task, createOrLookupChildForTaskBlock(task))
        case EXECUTED => become(executed(task.getId, task))
        case STOPPED => become(stopped(task.getId, task, createOrLookupChildForTaskBlock(task)))
        case FAILED => become(failed(task.getId, task, createOrLookupChildForTaskBlock(task)))
        case ABORTED => become(aborted(task.getId, task, createOrLookupChildForTaskBlock(task)))
        case _ => sender ! Failure(new IllegalStateException(s"Cannot recover a task which is in state [${task.state}]."))
      }
      TaskRegistryExtension(system).register(task)
      sender ! Registered(task.getId)
    case m @ _ => sender ! Failure(new IllegalStateException(s"Wrong command [$m] in UNREGISTERED state."))
  }

  def registerStateListeners(task: Task) {
    def registerStateListener(child: ActorRef) {
      context.system.eventStream.subscribe(child, classOf[TaskStateEvent])
      context.system.eventStream.subscribe(child, classOf[StepStateEvent])
    }

    val child: ActorRef = createChild(StateChangeEventListenerActor.props(task.getId), "state-listener")
    registerStateListener(child)
    if (task.getSpecification.isRecoverable) {
      val child: ActorRef = createChild(TaskRecoveryActor.props(task.getId, TaskerSettings(system).recoveryDir), "recovery-listener")
      registerStateListener(child)
    }
  }

  private[this] def calculateDelay(time: DateTime) : FiniteDuration = FiniteDuration((DateTime.now to time).millis, MILLISECONDS)

  def pending(taskId: TaskId, task: Task): Actor.Receive = {
   case Enqueue(`taskId`) =>
      doEnqueue(taskId, task, createOrLookupChildForTaskBlock(task))
    case Schedule(`taskId`, scheduleAt) =>
      doSchedule(taskId, task, scheduleAt, createOrLookupChildForTaskBlock(task))
    case Cancel(`taskId`, archiveActor, notificationActor) =>
      doCancelWhenPending(taskId, task, notificationActor)
    case GetTask(`taskId`) => sender ! task
    case m @ _ => sender ! Failure(new IllegalStateException(s"Wrong command [$m] for task [${task.getId}] in state [${task.state}]."))
  }

  private[this] def doEnqueue(taskId: TaskId, task: Task, blockActor: ActorRef) {
    log.info(s"Received [Enqueue] message for task [${task.getId}]")
    updateStateAndNotify(task, QUEUED)
    become(queued(taskId, task, blockActor))
    self ! Start(taskId)
  }

  def queued(taskId: TaskId, task: Task, blockActor: ActorRef): Actor.Receive = {
    case Start(`taskId`) =>
      doStart(taskId, task, blockActor)
    case m @ _ => sender ! Failure(new IllegalStateException(s"Wrong command [$m] for task [${task.getId}] in state [${task.state}]."))
  }

  private[this] def doStart(taskId: TaskId, task: Task, blockActor: ActorRef) {
    log.info(s"Received [Start] message for task [${task.getId}]")
    task.recordStart
    updateStateAndNotify(task, EXECUTING)
    become(executing(taskId, task, blockActor))
    blockActor ! Start(taskId)
  }

  private[this] def doSchedule(taskId: TaskId, task: Task, scheduleAt: DateTime, blockActor: ActorRef) {
    log.info(s"Received [Schedule] message for task [${task.getId}]")
    task.setScheduledDate(scheduleAt)
    val scheduledTask = updateStateAndNotify(task, SCHEDULED)
    val delay: FiniteDuration = calculateDelay(scheduleAt)
    log.info(s"Going to schedule task [$taskId] at [${scheduleAt.toString("yyyy-MM-dd hh:mm:ss Z")}] which is [$delay] from now")
    if (delay.toMinutes < 0) {
      doEnqueue(taskId, task, blockActor)
    } else {
      val scheduleHandle: Cancellable = context.system.scheduler.scheduleOnce(delay, self, Enqueue(taskId))
      become(scheduled(taskId, scheduledTask, scheduleHandle, blockActor))
    }
  }

  def doCancelWhenPending(taskId: TaskId, task: Task, notificationActor: ActorRef) {
    log.info(s"Received [Cancel] message for task [${task.getId}]")
    context.system.eventStream.subscribe(self, classOf[TaskStateEventHandled])
    notifyTaskDone(updateStateAndNotify(task, CANCELLED))
    TaskRegistryExtension(system).deleteTask(task.getId)
    var handledMessagesCounter = if (task.getSpecification.isRecoverable) 2 else 1
    become({
      case TaskStateEventHandled(`taskId`, _, CANCELLED) if handledMessagesCounter == 1 =>
        notificationActor ! Cancelled(taskId)
        context.system.eventStream.unsubscribe(self)
      case TaskStateEventHandled(`taskId`, _, CANCELLED) => handledMessagesCounter -= 1
      case _ =>
    })
  }

  def scheduled(taskId: TaskId, task: Task, scheduleHandle: Cancellable, blockActor: ActorRef): Actor.Receive = {
    case Enqueue(`taskId`) =>
      scheduleHandle.cancel()
      doEnqueue(taskId, task, blockActor)
    case Schedule(`taskId`, scheduleAt) =>
      scheduleHandle.cancel()
      doSchedule(taskId, task, scheduleAt, blockActor)
    case ScheduledStart(`taskId`) =>
      doEnqueue(taskId, task, blockActor)
    case Cancel(`taskId`, archiveActor, notificationActor) if task.getStartDate == null =>
      scheduleHandle.cancel()
      doCancelWhenPending(taskId, task, notificationActor)
    case Cancel(`taskId`, archiveActor, notificationActor) if task.getStartDate != null =>
      scheduleHandle.cancel()
      doCancel(taskId, task, archiveActor, notificationActor)
    case _ => throw new IllegalStateException()
  }

  def executing(taskId: TaskId, task: Task, blockActor: ActorRef): Actor.Receive = {
    case BlockStateChanged(`taskId`, block, oldState, state) =>
      state match {
        case BlockState.PENDING | BlockState.EXECUTING =>
        case BlockState.EXECUTED => become(executed(taskId, updateStateAndNotify(task, EXECUTED)))
        case BlockState.FAILING => become(failing(taskId, updateStateAndNotify(task, FAILING), blockActor))
        case BlockState.FAILED => become(failed(taskId, updateStateAndNotify(task, FAILED), blockActor))
        case BlockState.STOPPING => become(stopping(taskId, updateStateAndNotify(task, STOPPING), blockActor))
        case BlockState.STOPPED => become(stopped(taskId, updateStateAndNotify(task, STOPPED), blockActor))
        case BlockState.ABORTING => become(aborting(taskId, updateStateAndNotify(task, ABORTING), blockActor))
        case BlockState.ABORTED => become(aborted(taskId, updateStateAndNotify(task, ABORTED), blockActor))
      }
    case s @ Stop(`taskId`) =>
      log.debug(s"Received [$s], now stopping execution")
      blockActor ! s
    case m @ Abort(`taskId`) =>
      log.debug(s"Received [$m], now aborting execution")
      blockActor ! m
    case GetTask(`taskId`) => sender ! task
    case m @ _ => sender ! Failure(new IllegalStateException(s"Wrong command [$m] for task [${task.getId}] in state [${task.state}]."))
  }

  def executed(taskId: TaskId, task: Task): Actor.Receive = {
    case BlockDone(`taskId`, block) =>
      task.recordCompletion
      notifyTaskDone(task)
    case ArchiveTask(`taskId`, archiveActor: ActorRef, notificationActor: ActorRef) =>
      log.info(s"Received [Archive] message for task [${task.getId}]")
      context.system.eventStream.subscribe(self, classOf[TaskStateEventHandled])
      val doneTask: Task = updateStateAndNotify(task, DONE)
      TaskRegistryExtension(system).deleteTask(doneTask.getId)
      become(waitForStateHandledThenArchive(taskId, task, DONE, archiveActor, notificationActor, if (task.getSpecification.isRecoverable) 2 else 1))
    case GetTask(`taskId`) => sender ! task
    case m @ _ => sender ! Failure(new IllegalStateException(s"Wrong command [$m] for task [${task.getId}] in state [${task.state}]."))
  }

  def archiving(taskId: TaskId, task: Task, notificationActor: ActorRef): Actor.Receive = {
    case Archived(`taskId`) =>
      notifyTaskDone(task)
      task.getState2 match {
        case DONE => notificationActor ! Archived(taskId)
        case CANCELLED => notificationActor ! Cancelled(taskId)
        case _ =>
      }
      log.debug(s"Done with task [$taskId], now seppuku!")
      self ! PoisonPill
    case fta @ FailedToArchive(`taskId`, exception) =>
      task.getState2 match {
        case DONE => 
          updateStateAndNotify(task, EXECUTED)
          become(executed(taskId, task))
        case CANCELLED => 
          updateStateAndNotify(task, FAILED)
          become(failed(taskId, task, createOrLookupChildForTaskBlock(task)))
        case _ =>
      }      
      notifyTaskDone(task)
      notificationActor ! fta
    case _ =>
  }


  def failing(taskId: TaskId, task: Task, blockActor: ActorRef): Actor.Receive = {
    case BlockStateChanged(`taskId`, block, oldState, state) =>
      state match {
        case BlockState.PENDING | BlockState.EXECUTING | BlockState.EXECUTED | BlockState.FAILING | BlockState.STOPPING | BlockState.STOPPED =>
        case BlockState.FAILED => become(failed(taskId, updateStateAndNotify(task, FAILED), blockActor))
        case BlockState.ABORTING => become(aborting(taskId, updateStateAndNotify(task, ABORTING), blockActor))
        case BlockState.ABORTED => become(aborted(taskId, updateStateAndNotify(task, ABORTED), blockActor))
      }
    case GetTask(`taskId`) => sender ! task
    case m @ _ => sender ! Failure(new IllegalStateException(s"Wrong command [$m] for task [${task.getId}] in state [${task.state}]."))
  }


  def failed(taskId: TaskId, task: Task, blockActor: ActorRef): Actor.Receive = {
    case BlockDone(`taskId`, block) =>
      task.recordFailure
      task.recordCompletion
      notifyTaskDone(task)
    case Enqueue(`taskId`) =>
      doEnqueue(taskId, task, blockActor)
    case Schedule(`taskId`, scheduleAt) =>
      doSchedule(taskId, task, scheduleAt, blockActor)
    case Cancel(`taskId`, archiveActor, notificationActor) =>
      doCancel(taskId, task, archiveActor, notificationActor)
    case GetTask(`taskId`) => sender ! task
    case m @ _ => sender ! Failure(new IllegalStateException(s"Wrong command [$m] for task [${task.getId}] in state [${task.state}]."))
  }

  def doCancel(taskId: TaskId, task: Task, archiveActor: ActorRef, notificationActor: ActorRef) {
    log.info(s"Received [Cancel] message for task [${task.getId}]")
    context.system.eventStream.subscribe(self, classOf[TaskStateEventHandled])
    notifyTaskDone(updateStateAndNotify(task, CANCELLED))
    TaskRegistryExtension(system).deleteTask(task.getId)
    become(waitForStateHandledThenArchive(taskId, task, CANCELLED, archiveActor, notificationActor, if (task.getSpecification.isRecoverable) 2 else 1))
  }

  def waitForStateHandledThenArchive(taskId: TaskId, task: Task, state: TaskExecutionState2, archiveActor: ActorRef, notificationActor: ActorRef, handledMessages: Int): Actor.Receive = {
    case TaskStateEventHandled(`taskId`, _, `state`) if handledMessages == 1 =>
      become(archiving(taskId, task, notificationActor))
      archiveActor ! SendToArchive(task, self)
    case TaskStateEventHandled(`taskId`, _, `state`) => become(waitForStateHandledThenArchive(taskId, task, state, archiveActor, notificationActor, handledMessages - 1))
    case _ =>
  }

  def stopping(taskId: TaskId, task: Task, blockActor: ActorRef): Actor.Receive = {
    case BlockStateChanged(`taskId`, block, oldState, state) =>
      state match {
        case BlockState.PENDING | BlockState.EXECUTING | BlockState.EXECUTED | BlockState.STOPPING =>
        case BlockState.FAILING => become(failing(taskId, updateStateAndNotify(task, FAILING), blockActor))
        case BlockState.FAILED => become(failed(taskId, updateStateAndNotify(task, FAILED), blockActor))
        case BlockState.STOPPED => become(stopped(taskId, updateStateAndNotify(task, STOPPED), blockActor))
        case BlockState.ABORTING => become(aborting(taskId, updateStateAndNotify(task, ABORTING), blockActor))
        case BlockState.ABORTED => become(aborted(taskId, updateStateAndNotify(task, ABORTED), blockActor))
      }
    case GetTask(`taskId`) => sender ! task
    case m @ _ => sender ! Failure(new IllegalStateException(s"Wrong command [$m] for task [${task.getId}] in state [${task.state}]."))
  }

  def stopped(taskId: TaskId, task: Task, blockActor: ActorRef): Actor.Receive = {
    case BlockDone(`taskId`, block) =>
      task.recordCompletion
      notifyTaskDone(task)
    case Enqueue(`taskId`) =>
      doEnqueue(taskId, task, blockActor)
    case Schedule(`taskId`, scheduleAt) =>
      doSchedule(taskId, task, scheduleAt, blockActor)
    case Cancel(`taskId`, archiveActor, notificationActor) =>
      log.info(s"Received [Cancel] message for task [${task.getId}]")
      context.system.eventStream.subscribe(self, classOf[TaskStateEventHandled])
      notifyTaskDone(updateStateAndNotify(task, CANCELLED))
      TaskRegistryExtension(system).deleteTask(task.getId)
      become(waitForStateHandledThenArchive(taskId, task, CANCELLED, archiveActor, notificationActor, if (task.getSpecification.isRecoverable) 2 else 1))
    case GetTask(`taskId`) => sender ! task
    case m @ _ => sender ! Failure(new IllegalStateException(s"Wrong command [$m] for task [${task.getId}] in state [${task.state}]."))
  }

  def aborting(taskId: TaskId, task: Task, blockActor: ActorRef): Actor.Receive = {
    case BlockStateChanged(`taskId`, block, oldState, state) =>
      state match {
        case BlockState.PENDING | BlockState.EXECUTING | BlockState.EXECUTED | BlockState.FAILING | BlockState.STOPPING | BlockState.STOPPED | BlockState.FAILED | BlockState.ABORTING =>
        case BlockState.ABORTED => become(aborted(taskId, updateStateAndNotify(task, ABORTED), blockActor))
      }
    case GetTask(`taskId`) => sender ! task
    case m @ _ => sender ! Failure(new IllegalStateException(s"Wrong command [$m] for task [${task.getId}] in state [${task.state}]."))
  }

  def aborted(taskId: TaskId, task: Task, blockActor: ActorRef): Actor.Receive = {
    case BlockDone(`taskId`, block) =>
      task.recordFailure
      task.recordCompletion
      notifyTaskDone(task)
    case Enqueue(`taskId`) =>
      doEnqueue(taskId, task, blockActor)
    case Schedule(`taskId`, scheduleAt) =>
      doSchedule(taskId, task, scheduleAt, blockActor)
    case Cancel(`taskId`, archiveActor, notificationActor) =>
      log.info(s"Received [Cancel] message for task [${task.getId}]")
      context.system.eventStream.subscribe(self, classOf[TaskStateEventHandled])
      notifyTaskDone(updateStateAndNotify(task, CANCELLED))
      TaskRegistryExtension(system).deleteTask(task.getId)
      become(waitForStateHandledThenArchive(taskId, task, CANCELLED, archiveActor, notificationActor, if (task.getSpecification.isRecoverable) 2 else 1))
    case GetTask(`taskId`) => sender ! task
    case m @ _ => sender ! Failure(new IllegalStateException(s"Wrong command [$m] for task [${task.getId}] in state [${task.state}]."))
  }

  def createOrLookupChildForTaskBlock(task: Task): ActorRef = child(task.getBlock.id) match {
    case Some(ref) => ref
    case None => task.getBlock match {
      case sb: StepContainingBlock[_] => createChild(StepBlockExecutingActor.props(task, sb, task.getContext), sb.id)
      case b: CompositeBlock => createChild(BlockExecutingActor.props(task, b, task.getContext), b.id)
    }
  }

  def notifyTaskDone(task: Task) {
    log.info(s"Task [${task.getId}] is completed with state [${task.getState2}]")
    context.system.eventStream.publish(TaskDone(task))
  }
}
