package com.xebialabs.deployit.engine.tasker

import akka.actor.{Props, Cancellable, Actor, ActorLogging}
import concurrent.duration._
import com.xebialabs.deployit.engine.api.execution.TaskState
import java.io.{IOException, FileOutputStream, ObjectOutputStream, File}
import java.lang.String
import javassist.util.proxy.ProxyObjectOutputStream
import com.google.common.io.Closeables
import language.postfixOps
import com.xebialabs.deployit.engine.tasker.TaskRecoveryActor.Write
import com.xebialabs.deployit.engine.tasker.StateChangeEventListenerActor.{StepStateEvent, TaskStateEvent}
import com.xebialabs.deployit.engine.tasker.messages.TaskStateEventHandled

object TaskRecoveryActor {
  def props(taskId: TaskId, recoveryDir: File) = Props(classOf[TaskRecoveryActor], taskId, recoveryDir)

  case class Write(task: Task)
}

class TaskRecoveryActor(taskId: TaskId, recoveryDir: File) extends Actor with ActorLogging {


  import context._
  if (!recoveryDir.exists()) recoveryDir.mkdirs()

  def receive: Actor.Receive = {
    case e @ TaskStateEvent(`taskId`, _, _, state) if !state.isFinal =>
      log.debug(s"Received [$e]")
      become(writeScheduled(context.system.scheduler.scheduleOnce(1 second, self, Write(e.t))))
      log.debug(s"Scheduled Recovery writer for [$taskId] for 1 second from now")
    case e @ StepStateEvent(`taskId`, _, _, _, _, _, _) =>
      log.debug(s"Received [$e]")
      become(writeScheduled(context.system.scheduler.scheduleOnce(1 second, self, Write(e.t))))
      log.debug(s"Scheduled Recovery writer for [$taskId] for 1 second from now")
    case e @ TaskStateEvent(`taskId`, _, oldState, state) if state.isFinal =>
      log.debug(s"Cleaning up recovery file for [$taskId]")
      getFile(taskId).delete()
      context.system.eventStream.publish(TaskStateEventHandled(taskId, oldState, state))
  }

  def writeScheduled(token: Cancellable): Actor.Receive = {
    case e @ TaskStateEvent(`taskId`, _, oldState, state) if state.isFinal =>
      token.cancel()
      log.debug(s"Canceled writing recovery file for [$taskId]")
      getFile(taskId).delete()
      context.system.eventStream.publish(TaskStateEventHandled(taskId, oldState, state))
    case Write(task) =>
      log.debug(s"Writing recovery file for: [$task]")
      writeRecovery(task)
      become(receive)
    case e @ StepStateEvent =>
  }

  private def getFile(id: String): File = {
    new File(recoveryDir, s"$id.task")
  }

  private def writeRecovery(task: TaskState) {
    val id: String = task.getId
    var os: ObjectOutputStream = null
    try {
      os = new ProxyObjectOutputStream(new FileOutputStream(getFile(id)))
      os.writeObject(task)
    } catch {
      case e: IOException =>
        log.error(e, s"Could not write recovery file for [$id]")
    }
    finally {
      Closeables.closeQuietly(os)
    }
  }
}
