package com.xebialabs.deployit.engine.tasker

import akka.actor.{Props, Actor, ActorLogging}
import com.xebialabs.deployit.engine.spi.execution.{StepExecutionStateEvent, TaskExecutionStateEvent}
import com.xebialabs.deployit.engine.api.execution._
import scala.util.Try
import akka.actor.Status.{Failure, Success}
import com.xebialabs.deployit.engine.tasker.messages.{StepStateEventHandled, TaskStateEventHandled}
import com.xebialabs.deployit.engine.tasker.StateChangeEventListenerActor.{StepStateEvent, TaskStateEvent}

object StateChangeEventListenerActor {
  def props(taskId: TaskId) = Props(classOf[StateChangeEventListenerActor], taskId)
  
    case class TaskStateEvent(taskId: TaskId, t: Task, prev: TaskExecutionState, curr: TaskExecutionState) extends TaskExecutionStateEvent {

      def previousState(): TaskExecutionState = prev

      def currentState(): TaskExecutionState = curr

      def task(): TaskWithSteps = t
    }
    case class StepStateEvent(taskId: TaskId, stepId: StepId, t: Task, s: TaskStep, prev: StepExecutionState, curr: StepExecutionState, ctx: Option[TaskExecutionContext]) extends StepExecutionStateEvent {
      def previousState(): StepExecutionState = prev

      def currentState(): StepExecutionState = curr

      def step(): StepState = s

      def task(): TaskState = t
    }
}

class StateChangeEventListenerActor(taskId: TaskId) extends Actor with ActorLogging {
  def receive: Actor.Receive = {
    case e @ TaskStateEvent(`taskId`, _, _, _) => {
      log.debug(s"Received [$e]")
      e.t.getContext.allListeners.foreach { l =>
        log.debug(s"Notifying listeners [$l] of [$e]")
        Try(l.taskStateChanged(e))
      }
      context.system.eventStream.publish(TaskStateEventHandled(taskId, e.prev, e.curr))
    }
    case e @ StepStateEvent(`taskId`, stepId, _, _, _, _, _) => {
      log.debug(s"Received [$e]")
      e.ctx.getOrElse(e.t.context).listeners.foreach { l =>
        log.debug(s"Notifying listeners [$l] of [$e]")
        Try(l.stepStateChanged(e))
      }
      context.system.eventStream.publish(StepStateEventHandled(taskId, stepId, e.prev, e.curr))
    }
  }
  
  def wrapException(f: () => Unit) = Try(f()).map(u => Success).recover({case ex => Failure(ex)}).get
}
