package com.xebialabs.deployit.deployment.planner;

import com.google.common.collect.ArrayListMultimap;
import com.google.common.collect.ListMultimap;

import com.xebialabs.deployit.plugin.api.deployment.planning.*;
import com.xebialabs.deployit.plugin.api.deployment.specification.Delta;
import com.xebialabs.deployit.plugin.api.deployment.specification.DeltaSpecification;
import com.xebialabs.deployit.plugin.api.deployment.specification.Deltas;
import com.xebialabs.deployit.plugin.api.deployment.specification.Operation;
import com.xebialabs.deployit.plugin.api.execution.Step;
import com.xebialabs.deployit.plugin.api.services.Repository;

import nl.javadude.scannit.Scannit;

import java.lang.annotation.Annotation;
import java.lang.reflect.Method;
import java.util.*;

import static com.google.common.base.Preconditions.checkArgument;
import static com.google.common.collect.Lists.newArrayList;

public class DeploymentPlannerFactory {

    public static final Comparator<Method> PROCESSOR_COMPARATOR = new Comparator<Method>() {
        @Override
        public int compare(Method m1, Method m2) {
            String c1 = m1.getDeclaringClass().getSimpleName();
            String c2 = m2.getDeclaringClass().getSimpleName();

            int o1;
            int o2;
            if (m1.isAnnotationPresent(PrePlanProcessor.class)) {
                o1 = m1.getAnnotation(PrePlanProcessor.class).order();
                o2 = m2.getAnnotation(PrePlanProcessor.class).order();
            } else {
                o1 = m1.getAnnotation(PostPlanProcessor.class).order();
                o2 = m2.getAnnotation(PostPlanProcessor.class).order();
            }

            if (o1 != o2) {
                return Integer.valueOf(o1).compareTo(o2);
            }

            if (!c1.equals(c2)) {
                return c1.compareTo(c2);
            }

            return m1.getName().compareTo(m2.getName());
        }
    };

    public DeploymentPlanner planner() {

        ListMultimap<Operation, Method> deployedContributors = registerDeployedContributors();
        Set<Method> contributors = registerContributors();
        Scannit instance = Scannit.getInstance();
        List<Method> preProcessors = orderProcessors(checkProcessors(instance.getMethodsAnnotatedWith(PrePlanProcessor.class)));
        List<Method> postProcessors = orderProcessors(checkProcessors(instance.getMethodsAnnotatedWith(PostPlanProcessor.class)));

        return new DeploymentPlanner.DeploymentPlannerBuilder()
                .preProcessors(preProcessors)
                .postProcessors(postProcessors)
                .typeContributors(deployedContributors)
                .contributors(contributors)
                .build();
    }

    private List<Method> orderProcessors(Set<Method> methods) {
        ArrayList<Method> list = newArrayList(methods);
        Collections.sort(list, PROCESSOR_COMPARATOR);
        return list;
    }

    private ListMultimap<Operation, Method> registerDeployedContributors() {
        ListMultimap<Operation, Method> deployedContributorMap = ArrayListMultimap.create();

        registerDeployedContributors(Create.class, Operation.CREATE, deployedContributorMap);
        registerDeployedContributors(Modify.class, Operation.MODIFY, deployedContributorMap);
        registerDeployedContributors(Destroy.class, Operation.DESTROY, deployedContributorMap);
        registerDeployedContributors(Noop.class, Operation.NOOP, deployedContributorMap);

        return deployedContributorMap;
    }

    private void registerDeployedContributors(Class<? extends Annotation> annotation, Operation operation, ListMultimap<Operation, Method> deployedContributorMap) {
        Set<Method> typeContributors = new TreeSet<Method>(new HierarchyClassMethodNameComparator());
        typeContributors.addAll(checkDeployedContributors(Scannit.getInstance().getMethodsAnnotatedWith(annotation)));
        for (Method typeContributor : typeContributors) {
            deployedContributorMap.put(operation, typeContributor);
        }
    }

    private Set<Method> registerContributors() {
        Set<Method> contributors = new TreeSet<Method>(new HierarchyClassMethodNameComparator());
        contributors.addAll(checkContributors(Scannit.getInstance().getMethodsAnnotatedWith(Contributor.class)));
        return contributors;
    }

    private Set<Method> checkDeployedContributors(Set<Method> deployedContributors) {
        for (Method c : deployedContributors) {
            checkArgument(c.getReturnType().equals(void.class), "DeployedContributor %s should have void return type.", c);
            Class<?>[] parameterTypes = c.getParameterTypes();
            checkArgument(parameterTypes.length <= 2 && parameterTypes.length >= 1, "DeployedContributor %s should take 1 or 2 parameters.", c);
            checkArgument(parameterTypes[0].equals(DeploymentPlanningContext.class), "DeployedContributor %s should take %s as first parameter.", c, DeploymentPlanningContext.class);
            if (parameterTypes.length == 2) {
                checkArgument(parameterTypes[1].equals(Delta.class), "DeployedContributor %s should take %s as first parameter.", c, Delta.class);
            }
        }
        return deployedContributors;
    }

    private Set<Method> checkContributors(Set<Method> contributors) {
        for (Method contributor : contributors) {
            checkArgument(contributor.getReturnType().equals(void.class), "Contributor %s should have void return type.", contributor);
            Class<?>[] parameterTypes = contributor.getParameterTypes();
            checkArgument(parameterTypes.length == 2, "Contributor %s should take 2 parameters.", contributor);
            checkArgument(parameterTypes[0].equals(Deltas.class), "Contributor %s should take %s as first parameter.", contributor, Deltas.class);
            checkArgument(parameterTypes[1].equals(DeploymentPlanningContext.class), "Contributor %s should take %s as second parameter.", contributor, DeploymentPlanningContext.class);
        }
        return contributors;
    }

    private Set<Method> checkProcessors(Set<Method> processors) {
        for (Method processor : processors) {
            Class<?> returnType = processor.getReturnType();
            checkArgument(returnType.equals(Step.class) || returnType.equals(com.xebialabs.deployit.plugin.api.flow.Step.class) || returnType.equals(List.class), "Pre/Post processor %s should have a Step or List<Step> return type.", processor);
            Class<?>[] parameterTypes = processor.getParameterTypes();
            checkArgument(parameterTypes.length == 1, "Processor %s should take 1 parameter.", processor);
            checkArgument(parameterTypes[0].equals(DeltaSpecification.class), "Processor %s should take %s as first parameter.", processor, DeltaSpecification.class);
        }
        return processors;
    }

    /**
     * - First compare on the Class hierarchy, subclasses before superclasses
     * - Then on the class name
     * - Then on the method name
     */
    public static class HierarchyClassMethodNameComparator implements Comparator<Method> {

        @Override
        public int compare(Method method, Method method1) {
            Class<?> class1 = method.getDeclaringClass();
            Class<?> class2 = method1.getDeclaringClass();

            if (isSuperClass(class1, class2)) {
                return -1;
            } else if (isSuperClass(class2, class1)) {
                return 1;
            } else {
                String name1 = class1.getName();
                String name2 = class2.getName();
                int nameComparison = name1.compareTo(name2);
                if (nameComparison == 0) {
                    return method.getName().compareTo(method1.getName());
                } else {
                    return nameComparison;
                }
            }
        }

        boolean isSuperClass(Class<?> c1, Class<?> c2) {
            return c1.isAssignableFrom(c2) && !c1.getName().equals(c2.getName());
        }
    }
}
