package com.xebialabs.deployit.deployment.rules;

import java.io.IOException;
import java.io.InputStream;
import java.net.URL;
import java.util.List;
import org.jdom2.*;
import org.jdom2.input.SAXBuilder;
import org.jdom2.input.sax.XMLReaderXSDFactory;

import com.google.common.io.Closeables;

import static com.google.common.collect.Lists.newArrayList;
import static java.lang.String.format;
import static java.lang.Thread.currentThread;

public class XmlRulesParser {

    public static final Namespace RULES_NAMESPACE = Namespace.getNamespace("http://www.xebialabs.com/deployit/rules");

    public void parseRules(URL rulesFile, RuleStore store) {
        SAXBuilder b;
        try {
            XMLReaderXSDFactory xmlReaderXSDFactory = new XMLReaderXSDFactory(currentThread().getContextClassLoader().getResource("rules.xsd"));
            b = new SAXBuilder(xmlReaderXSDFactory);
        } catch (JDOMException e) {
            throw new IllegalStateException("Could not find the 'rules.xsd' XSD file", e);
        }
        InputStream in = null;
        try {
            in = rulesFile.openStream();
            Document document = b.build(in);
            Element rootElement = document.getRootElement();
            List<Element> rulesElements = newArrayList();
            List<Element> disableRulesElements = newArrayList();
            collectElements(rootElement, rulesElements, disableRulesElements);
            parseElements(rulesElements, disableRulesElements, store);
        } catch (IOException e) {
            e.printStackTrace(); // To change body of catch statement use File | Settings | File Templates.
        } catch (JDOMException e) {
            e.printStackTrace(); // To change body of catch statement use File | Settings | File Templates.
        } finally {
            Closeables.closeQuietly(in);
        }
    }

    private void collectElements(Element rootElement, List<Element> rulesElements, List<Element> disableRulesElements) {
        for (Element element : rootElement.getChildren()) {
            if (element.getName().equals("rule")) {
                rulesElements.add(element);
            } else if (element.getName().equals("disable-rule")) {
                disableRulesElements.add(element);
            }
        }
    }

    private void parseElements(List<Element> rulesElements, List<Element> disableRulesElements, RuleStore store) {
        for (Element rulesElement : rulesElements) {
            parseRule(rulesElement, store);
        }
        for (Element disableRulesElement : disableRulesElements) {
            parseRuleDisable(disableRulesElement, store);
        }
    }

    private void parseRule(Element ruleElement, RuleStore store) {
        String name = ruleElement.getAttributeValue("name");
        String scopeName = ruleElement.getAttributeValue("scope");
        Scope scope = Scope.getScope(scopeName);
        ScriptRule scriptRule = new ScriptRule(name, scope);
        if (ruleElement.getChild("script", RULES_NAMESPACE) != null) {
            scriptRule.setScript(ruleElement.getChildText("script", RULES_NAMESPACE));
        }
        store.registerRule(scriptRule);
    }

    private void parseRuleDisable(Element element, RuleStore store) {
        String name = element.getAttributeValue("name");
        Rule rule = store.getRule(name);
        if (rule != null) {
            rule.disable();
        } else {
            throw new IllegalArgumentException(format("Rule [%s] has not been defined, and cannot be disabled", name));
        }
    }

}
