package com.xebialabs.deployit.deployment.stager

import com.xebialabs.deployit.deployment.planner._
import grizzled.slf4j.Logging
import collection.convert.wrapAll._
import com.xebialabs.deployit.plugin.api.flow._
import com.xebialabs.overthere.{OverthereConnection, OverthereFile}
import com.xebialabs.deployit.plugin.api.udm.artifact.Artifact
import scala.util.Try
import scala.collection.mutable.ListBuffer
import com.xebialabs.deployit.deployment.stager.DeploymentStager.{StagingFile, _StagingContext}
import scala.Some
import scala.collection.mutable

class DeploymentStager(taskId: String) extends Stager with Logging {

  private[this] val stagingContext = new _StagingContext(taskId)

  // Duck typing, Ric forced me to rename this from Duck to HasStagingFile ;-)
  type HasStagingTarget = { def getStagingTarget: StagingTarget }

  def stage(plan: Plan): Plan = {
    logger.info(s"Staging artifacts for plan: [${plan.getDescription}]")
    doStage(plan)
    if (stagingContext.stagingSteps.isEmpty) {
      plan
    } else {
      def stepsToPlan(steps : Iterable[Step with HasStagingTarget], descriptionPrefix: String): List[StepPlan] = {
        steps.groupBy(_.getStagingTarget).map({ case (t, s) => new StepPlan(s"$descriptionPrefix ${t.getName}", s, plan.getListeners) }).toList
      }

      val stagingPlan = new ParallelPlan("Stage artifacts", stepsToPlan(stagingContext.stagingSteps, "Staging to"), plan.getListeners)
      val cleanupPlan = new ParallelPlan("Clean up staged artifacts", stepsToPlan(stagingContext.cleanupHosts.map(t => new StagedFileCleaningStep(t, taskId)), "Clean up staged files on"), plan.getListeners)
      val planWithStaging = new SerialPlan(plan.getDescription, List(stagingPlan, plan, cleanupPlan), plan.getListeners)
      planWithStaging
    }
  }

  def doStage(plan: Plan) {
    logger.debug(s"Staging for [${plan.getClass.getSimpleName}(${plan.getDescription})]")
    plan match {
      case cp: CompositePlan => cp.getSubPlans.foreach(doStage)
      case sp: StepPlan => doStage(sp)
    }
  }

  def doStage(stepPlan: StepPlan) {
    stepPlan.getSteps.withFilter(_.isInstanceOf[StageableStep]).foreach(step => doStage(step.asInstanceOf[StageableStep]))
  }

  def doStage(step: StageableStep) {
    logger.debug(s"Preparing stage of artifacts for step [${step.getDescription}]")
    step.requestStaging(stagingContext)
  }
}

object DeploymentStager extends Logging {

  private[stager] class JustInTimeFile(source: Artifact) extends StagedFile {
    def get(c: OverthereConnection, ctx: ExecutionContext): OverthereFile = {
      logger.debug(s"Creating temporary file for artifact [${source.getName}]")
      val uploadedFileArtifact: OverthereFile = c.getTempFile(source.getFile.getName)

      ctx.logOutput(s"Uploading artifact [${uploadedFileArtifact.getPath}]")
      source.getFile.copyTo(uploadedFileArtifact)

      logger.debug(s"Uploaded artifact [${uploadedFileArtifact.getPath}]")
      uploadedFileArtifact
    }
  }

  private[stager] class StagingFile(source: Artifact, target: StagingTarget, taskId: String) extends StagedFile {

    def getSource = source

    def getTarget = target

    def getTaskId = taskId

    var stagedFilePath: Option[String] = None

    def get(notUsed: OverthereConnection, ctx: ExecutionContext): OverthereFile = {
      stagedFilePath.map(target.getConnection.getFile).getOrElse({
        val file = stageFile(ctx)
        stagedFilePath = Some(file.getPath)
        file
      })
    }

    def stageFile(ctx: ExecutionContext): OverthereFile = {
      ctx.logOutput(s"Staging file [${source.getFile}] to [${target.getConnection}]")
      val connection: OverthereConnection = target.getConnection
      try {
        val stagingDir: OverthereFile = connection.getFile(target.getStagingDirectoryPath)

        if (!stagingDir.exists()) {
          ctx.logOutput(s"Creating staging directory [$stagingDir]")
          stagingDir.mkdirs()
        }

        val taskStagingDir: OverthereFile = stagingDir.getFile(taskId)
        if (!taskStagingDir.exists()) {
          ctx.logOutput(s"Creating staging directory for task [$taskStagingDir]")
          taskStagingDir.mkdir()
        }

        var counter = 0
        def tryCreateLoop(dir: OverthereFile): OverthereFile = Try({
          logger.debug(s"Attempting to create artifact staging dir [$dir]")
          dir.mkdir()
          dir
        }).getOrElse({counter += 1; tryCreateLoop(taskStagingDir.getFile(s"${source.getName}-$counter"))})

        val artifactStagingDir: OverthereFile = tryCreateLoop(taskStagingDir.getFile(source.getName))
        ctx.logOutput(s"Successfully created artifact staging dir [$artifactStagingDir]")

        val stagedFile: OverthereFile = artifactStagingDir.getFile(source.getFile.getName)
        ctx.logOutput(s"Uploading [${source.getFile}] to [$stagedFile]")
        source.getFile.copyTo(stagedFile)
        logger.debug(s"Staged file [$stagedFile]")

        stagedFile
      } catch {
        case ex: Exception =>
          ctx.logError(s"Failed to upload [${source.getFile}] to [${target.getConnection}]")
          throw ex
      }
    }
  }

  private[stager] class _StagingContext(taskId: String) extends StagingContext {
    import collection.mutable.{Set, HashSet}
    val stagingSteps: ListBuffer[StagingStep] = new ListBuffer[StagingStep]()
    val cleanupHosts: mutable.Set[StagingTarget] = new mutable.HashSet[StagingTarget]()

    def stageArtifact(artifact: Artifact, target: StagingTarget): StagedFile = {
      if (Option(target.getStagingDirectoryPath).filterNot(_.trim.isEmpty).isEmpty) {
        new JustInTimeFile(artifact)
      } else {
        val file: StagingFile = new StagingFile(artifact, target, taskId)
        stagingSteps += new StagingStep(file)
        cleanupHosts += target
        file
      }
    }
  }

}
