package com.xebialabs.deployit.deployment
package rules

import java.lang.reflect.{InvocationTargetException, Field}

import com.xebialabs.deployit.plugin.api.flow.Step
import com.xebialabs.deployit.plugin.api.rules.{StepMetadata, RulePostConstruct, StepPostConstructContext}
import grizzled.slf4j.Logging
import scala.collection.convert.wrapAll._
import java.util

class StepFactory(stepRegistry: StepRegistry, stepPostConstructContext: StepPostConstructContext) extends Logging {

  def step(stepData: StepData) = {
    val stepName: String = stepData.stepName
    val stepParameters: Map[String, Any] = stepData.stepParameters
    trace(s"Creating step $stepName with parameters $stepParameters")
    createStepByDescriptor(stepRegistry.getStepDescriptor(stepName), stepParameters)
  }

  private def createStepByDescriptor(stepDescriptor: StepDescriptor, stepParameterValues: Map[String, Any]): Step = {
    val step = stepDescriptor.implementationClass.newInstance()
    stepParameterValues.foreach { case (name, value) =>
      val parameter = stepDescriptor.parameters.getOrElse(name, throw new IllegalArgumentException(s"Parameter '$name' could not be found for step ${stepDescriptor.name}"))
      setStepParameterValue(step, parameter, value)
    }
    doPostConstruction(step)
    validateAllRequiredParametersAreSet(stepDescriptor, step)
    step
  }

  private def setStepParameterValue(step: Step, descriptor: StepParameterDescriptor, value: Any) {
    val parameterName: String = descriptor.field.getName
    val field = descriptor.field
    val javaHappyValue = descriptor match {
      case d if d.isFieldOfType[util.Map[_, _]] && value.isInstanceOf[Map[_, _]] => new util.HashMap[String, Any](value.asInstanceOf[Map[String, Any]])
      case d if d.isFieldOfType[util.List[_]] && value.isInstanceOf[List[_]]=> new util.ArrayList[Any](value.asInstanceOf[List[Any]])
      case _ => value
    }
    field.setAccessible(true)
    try{
      field.set(step, javaHappyValue)
    } catch {
      case ex: Exception => throw new IllegalArgumentException(s"Value of type ${Option(javaHappyValue).map(_.getClass.getName).getOrElse("[null]")} cannot be assigned to parameter '$parameterName' of type ${field.getType.getName}")
    }
  }

  private def validateAllRequiredParametersAreSet(stepDescriptor: StepDescriptor, step: Step) {

    def notNullAndBlank(value: Any) = value != null && value.toString != ""

    def getValue(anyRef: AnyRef, field: Field) = {
      field.setAccessible(true)
      field.get(anyRef)
    }

    stepDescriptor.parameters.filter(_._2.required).foreach { case (paramName, paramDescriptor) =>
      require(notNullAndBlank(getValue(step, paramDescriptor.field)),
        s"Cannot create step '${stepDescriptor.name}' since required parameter '$paramName' is not specified")
    }
  }

  private def doPostConstruction(step: Step) = {
    trace(s"post-constructing [${step.getClass.getName}]")
    getClassChain(step.getClass)
      .flatMap(_.getDeclaredMethods)
      .filter(m => m.getAnnotation(classOf[RulePostConstruct]) != null)
      .sortBy(_.getName)
      .foreach(m => {
      debug(s"${step.getClass.getSimpleName} invoking [${m.getName}]")
      m.setAccessible(true)
      try {
        m.invoke(step, stepPostConstructContext)
      } catch {
        case e: InvocationTargetException => throw PlannerException(s"Error while post-constructing [${step.getClass.getAnnotation(classOf[StepMetadata]).name()}] step: ${e.getTargetException.getMessage}", e)
      }
    })
    trace(s"done post-constructing [${step.getClass.getName}]")
  }
}

case class StepData(stepName: String, stepParameters: Map[String, Any] = Map.empty)
