package com.xebialabs.deployit.deployment.planner

import com.xebialabs.deployit.plugin.api.deployment.specification.DeltaSpecification
import com.xebialabs.deployit.plugin.api.services.Repository

import collection.convert.wrapAll._
import scala.collection.mutable
import com.xebialabs.deployit.deployment.planner.StepPlan.StepWithPlanningInfo

class OrderChunkingPlanner (wrappedPlanner: Planner) extends Planner with ChunkingPlannerBase {

  val threshold: Int = 30

  val orderDescr = Map(0 -> "Pre flight", 1 -> "Stop artifacts", 2 -> "Stop containers", 3 -> "Undeploy artifacts", 4 -> "Destroy resources",
                        5 -> "In between", 6 -> "Create resources", 7 -> "Deploy artifacts", 8 -> "Start containers", 9 -> "Start artifacts", 10 -> "Post flight")

  def plan(spec: DeltaSpecification, repository: Repository): Plan = {
    val complexPlan: Plan = wrappedPlanner.plan(spec, repository)
    chunkByOrder(complexPlan)
  }

  def chunkByOrder(plan: Plan): Plan = plan match {
    case pp: ParallelPlan => new ParallelPlan(pp.getDescription, pp.getSubPlans.map(chunkByOrder), pp.getListeners)
    case sp: SerialPlan => new SerialPlan(sp.getDescription, sp.getSubPlans.map(chunkByOrder), sp.getListeners)
    case stp: StepPlan if stp.getSteps.size() > threshold => val chunks = chunkByOrder(stp)
                                                             if(chunks.length <= 1) stp
                                                             else new SerialPlan(stp.getDescription, chunks, stp.getListeners)
    case stp: StepPlan => stp
  }

  def chunkByOrder(plan: StepPlan): List[StepPlan] = {
    plan.getStepsWithPlanningInfo
      .groupBy(_.getStep.getOrder / 10).toList
      .sortBy(_._1)
      .map(createStepPlanForOrderedChunks(_, plan))
  }

  def createStepPlanForOrderedChunks(b : (Int, mutable.Buffer[StepWithPlanningInfo]), plan: StepPlan): StepPlan = {
    b match {
      case (i, steps) => new StepPlan(s"Steps with order ${orderDescr.getOrElse(steps(0).getStep.getOrder / 10, "Undefined")}",
                                      steps,
                                      findCheckpoints(steps, plan.getCheckpoints.toList),
                                      plan.getListeners)
    }
  }

}
