package com.xebialabs.deployit.deployment.planner

import com.xebialabs.deployit.deployment.planner.PlanSugar._
import com.xebialabs.deployit.deployment.planner.StepPlan.StepWithPlanningInfo
import com.xebialabs.deployit.plugin.api.deployment.specification.DeltaSpecification
import com.xebialabs.deployit.plugin.api.flow.Step
import com.xebialabs.deployit.plugin.api.services.Repository
import com.xebialabs.deployit.plugin.satellite._
import com.xebialabs.xlplatform.satellite.{Satellite, SatelliteAware}

import scala.collection.convert.wrapAll._
import scala.collection.mutable.ListBuffer

class SatelliteDeploymentPlanner(wrappedPlanner: Planner) extends Planner with PlanSugar {

  private def pingStepDescription(satellites: Seq[Satellite]) =
    s"Checking satellite${if (satellites.size > 1) "s" else ""} connectivity"

  private def preparingPhaseDescription(satellites: Seq[Satellite]) =
    s"Preparing task on satellite${if (satellites.size > 1) "s" else ""}"

  private def preparingPlanDescription(satellite: Satellite) =
    s"Preparing task on satellite ${satellite.getName}"

  private def cleanUpPhaseDescription(satellites: Seq[Satellite]) =
    s"Cleaning up satellite${if (satellites.size > 1) "s" else ""}"

  override def plan(spec: DeltaSpecification, repository: Repository): PhasedPlan = {
    val satellitePlan = wrappedPlanner.plan(spec, repository)
    val promotedPlan = promotePlan(satellitePlan)

    val allSatellites = extractSatellitesFromPlans(promotedPlan)

    val planWithPingSteps = addPingSteps(promotedPlan, allSatellites)

    val preparePhase = addPreparePhase(planWithPingSteps, allSatellites)

    addCleanUpSatellitePhase(preparePhase, allSatellites)
  }

  private def addPingSteps(promotedPlan: PhasedPlan, allSatellites: List[Satellite]): PhasedPlan = {
    promotedPlan.copy(phases = promotedPlan.phases.map(addPingSteps(_, allSatellites)).toList)
  }

  private def addPingSteps(phase: PlanPhase, allSatellites: List[Satellite]): PlanPhase = {
    allSatellites match {
      case Nil => phase
      case satellites =>
        val pingSteps = satellites.map(PingSatelliteStep(_))
        val pingStepPlan = new StepPlan(pingStepDescription(allSatellites), pingSteps, phase.getListeners)
        phase.copy(plan = new SerialPlan(
          phase.plan.getDescription + s" with satellite${if (satellites.size > 1) "s" else ""} ${allSatellites.map(_.getName).sorted.mkString(", ")}",
          List(pingStepPlan, phase.plan),
          phase.getListeners))
    }
  }

  private def addPreparePhase(promotedPlan: PhasedPlan, allSatellites: List[Satellite]): PhasedPlan = {
    allSatellites match {
      case Nil =>
        promotedPlan

      case satellite :: Nil =>
        val checkStep = CheckPluginStep(satellite)
        val sendToSatelliteStep = SendToSatelliteStep(satellite)
        val prepareSatellitePlan = new StepPlan(preparingPlanDescription(satellite), List(checkStep, sendToSatelliteStep), promotedPlan.getListeners)

        val sendToSatellitePhase = new PlanPhase(prepareSatellitePlan, preparingPhaseDescription(allSatellites), promotedPlan.getListeners)
        promotedPlan.copy(phases = sendToSatellitePhase :: promotedPlan.phases.toList)

      case satellites =>
        val prepareSatellitePlans = satellites.map { sat =>
          val checkStep = CheckPluginStep(sat)
          val sendToSatelliteStep = SendToSatelliteStep(sat)
          new StepPlan(preparingPlanDescription(sat), List(checkStep, sendToSatelliteStep), promotedPlan.getListeners)
        }

        val preparingPlan = new ParallelPlan(
          preparingPhaseDescription(allSatellites) + s" ${allSatellites.map(_.getName).sorted.mkString(", ")}",
          prepareSatellitePlans,
          promotedPlan.getListeners)

        val sendToSatellitePhase = new PlanPhase(preparingPlan, preparingPhaseDescription(allSatellites), promotedPlan.getListeners)
        promotedPlan.copy(phases = sendToSatellitePhase :: promotedPlan.phases.toList)
    }
  }

  private def addCleanUpSatellitePhase(promotedPlan: PhasedPlan, allSatellites: List[Satellite]): PhasedPlan = {
    allSatellites match {
      case Nil =>
        promotedPlan

      case satellites =>
        val steps = satellites.map(CleanUpSatelliteStep(_))
        val cleanupPlan = new StepPlan(cleanUpPhaseDescription(allSatellites) + s" ${allSatellites.map(_.getName).sorted.mkString(", ")}", steps, promotedPlan.getListeners)
        val cleanUpSatellitePhase = new PlanPhase(cleanupPlan, cleanUpPhaseDescription(allSatellites), promotedPlan.getListeners, true)
        promotedPlan.copy(phases = promotedPlan.phases.toList :+ cleanUpSatellitePhase)
    }
  }

  private def extractSatellitesFromPlans(promotedPlan: PhasedPlan): List[Satellite] =
    promotedPlan.phases
      .map(_.plan)
      .flatMap(extractSteps)
      .collect {
      case step: SatelliteAware => step
    }.map(_.getSatellite)
      .filterNot(_ == null)
      .toSet.toList

  def promotePlan(satellitePlan: Plan): PhasedPlan = {
    satellitePlan match {
      case phasedPlan: PhasedPlan => promoteSatellite(phasedPlan)
      case planPhase: PlanPhase => new PhasedPlan(List(promoteSatellite(planPhase)), planPhase.getListeners)
      case plan: ExecutablePlan => new PhasedPlan(List(new PlanPhase(promoteSatellite(plan), plan.getDescription, plan.getListeners)), plan.getListeners)
    }
  }

  private def extractSteps(plan: ExecutablePlan): Seq[Step] = plan match {
    case stepPlan: StepPlan => stepPlan.getSteps
    case composite: CompositePlan => composite.getSubPlans.flatMap(extractSteps)
  }

  private def promoteSatellite(plan: ExecutablePlan): ExecutablePlan = {
    plan match {
      case stepPlan: StepPlan => promoteSatellite(stepPlan)
      case serialPlan: SerialPlan => promoteSatellite(serialPlan)
      case parallelPlan: ParallelPlan => promoteSatellite(parallelPlan)
    }
  }

  private def promoteSatellite(phasedPlan: PhasedPlan): PhasedPlan = {
    phasedPlan.copy(phases = phasedPlan.phases.map(promoteSatellite))
  }

  private def promoteSatellite(planPhase: PlanPhase): PlanPhase = {
    planPhase.copy(plan = promoteSatellite(planPhase.plan))
  }

  private def promoteSatellite(stepPlan: StepPlan): ExecutablePlan = {
    val result = stepPlan.getStepsWithPlanningInfo.foldLeft(List[(Satellite, ListBuffer[StepWithPlanningInfo])]()) {
      (tuples, step) =>

        val satOption = getSatelliteOrNull(step.getStep)

        tuples match {
          case Nil => List((satOption, ListBuffer(step)))
          case (currSat, buff) :: _ if currSat == satOption => buff.append(step)
            tuples
          case _ => (satOption, ListBuffer(step)) :: tuples
        }
    }
    buildStepPlanResult(result, stepPlan)
  }

  private def buildStepPlanResult(result: List[(Satellite, ListBuffer[StepWithPlanningInfo])], originalPlan: StepPlan): ExecutablePlan = {
    result match {
      case List((null, swpi)) => originalPlan
      case List((sat, swpi)) => originalPlan.copy(satellite = sat, description = originalPlan.getDescription + s" with satellite ${sat.getName}")
      case _ =>
        val reversedTuples: List[(Satellite, ListBuffer[StepWithPlanningInfo])] = result.reverse
        val stepPlans = reversedTuples.map {
          case (sat, swpi) if sat == null => originalPlan.copy(steps = swpi, satellite = sat)
          case (sat, swpi) => originalPlan.copy(originalPlan.getDescription + s" ${
            sat.getName
          }", swpi, findCheckpoints(swpi, originalPlan.getCheckpoints), satellite = sat)
        }

        originalPlan.toSerial(subPlans = stepPlans)
    }
  }

  private def promoteSatellite(plan: CompositePlan): ExecutablePlan = {
    val modifiedSubPlans = plan.getSubPlans.map(promoteSatellite)
    val satellites = modifiedSubPlans.map(_.satellite).toSet

    val promotedSatellite = if (satellites.size == 1) {
      satellites.head
    } else {
      null
    }

    plan.copy(subPlans = modifiedSubPlans, satellite = promotedSatellite)
  }

  private def getSatelliteOrNull(step: Step): Satellite = step match {
    case stepSat: SatelliteAware => stepSat.getSatellite
    case _ => null
  }
}
