package com.xebialabs.deployit.deployment.rules

import java.lang.reflect.{InvocationTargetException, Modifier, Method}
import com.xebialabs.deployit.plugin.api.deployment.specification.DeltaSpecification
import com.xebialabs.deployit.plugin.api.Deprecations._
import com.xebialabs.deployit.deployment.planner.StepAdapter
import com.xebialabs.deployit.booter.local.utils.ReflectionUtils._
import com.xebialabs.deployit.plugin.api.flow.Step
import com.xebialabs.deployit.plugin.api.deployment.execution.DeploymentStep
import com.xebialabs.deployit.plugin.api.rules.Scope
import com.xebialabs.deployit.plugin.api.udm.DeployedApplication

import collection.convert.wrapAll._

object JavaProcessorInvokerRule {
  def buildName(processor: Method) = s"${processor.getDeclaringClass.getName}.${processor.getName}"
}

import JavaProcessorInvokerRule._

class JavaProcessorInvokerRule(processor: Method, scope: Scope, order: Int) extends Rule(buildName(processor), scope, Option(order)) {

  require(scope.equals(Scope.POST_PLAN) || scope.equals(Scope.PRE_PLAN))

  override def canFire(scopedObject: AnyRef, context: RulePlanningContext): Boolean = scopedObject.isInstanceOf[DeltaSpecification] && correctAppliedDistribution(context).isInstanceOf[DeployedApplication]

  override def doFire(deltaSpec: AnyRef, context: RulePlanningContext): Unit = {
    val compatibleContext = context.backwardCompatible
    try {
      val stepObj = if (Modifier.isStatic(processor.getModifiers)) {
        Option(processor.invoke(null, deltaSpec))
      } else {
        deprecated("Non-static pre and post processors are considered deprecated, consider making [%s] static", processor)
        val contributorInstance: Any = processor.getDeclaringClass.newInstance
        Option(processor.invoke(contributorInstance, deltaSpec))
      }
      stepObj match {
        case Some(step: DeploymentStep)     => compatibleContext.addStep(StepAdapter.wrapIfNeeded(step))
        case Some(step: Step)               => compatibleContext.addStep(step)
        case Some(steps: java.util.List[_]) => compatibleContext.addSteps(steps.map(StepAdapter.wrapIfNeeded))
        case Some(x)                        => warn(s"Unexpected return type for processor $processor. Expected: [Step] or List[Step] found: ${x.getClass}")
        case None                           =>
      }
    } catch {
      case e: InvocationTargetException => throw handleInvocationTargetException(e, "")
      case e: Exception => throw e
    }
  }
}
