package com.xebialabs.deployit.deployment.planner

import com.xebialabs.deployit.deployment.planner.PlanSugar._
import com.xebialabs.deployit.deployment.planner.StepPlan.StepWithPlanningInfo
import com.xebialabs.deployit.plugin.api.deployment.specification.DeltaSpecification
import com.xebialabs.deployit.plugin.api.services.Repository

import scala.collection.convert.wrapAll._
import scala.collection.mutable

class SizeChunkingPlanner(wrappedPlanner: Planner) extends Planner with ChunkingPlannerBase with PlanSugar {

  val threshold: Int = 100

  def plan(spec: DeltaSpecification, repository: Repository): PhasedPlan = {
    val complexPlan = wrappedPlanner.plan(spec, repository)
    chunkBySize(complexPlan)
  }

  def chunkBySize(complexPlan: Plan): PhasedPlan = {
    complexPlan match {
      case phasedPlan: PhasedPlan => chunkBySize(phasedPlan)
      case planPhase: PlanPhase => new PhasedPlan(List(chunkBySize(planPhase)), planPhase.getListeners)
      case plan: ExecutablePlan => new PhasedPlan(List(new PlanPhase(chunkBySize(plan), plan.getDescription, plan.getListeners)), plan.getListeners)
    }
  }

  def chunkBySize(plan: ExecutablePlan): ExecutablePlan = plan match {
    case cp: CompositePlan => cp.copy(subPlans = cp.getSubPlans.map(chunkBySize))
    case stp: StepPlan if stp.getSteps.size() > threshold => stp.toSerial(subPlans = chunkBySize(stp))
    case stp: StepPlan => stp
  }

  def chunkBySize(phasedPlan: PhasedPlan): PhasedPlan = {
    phasedPlan.copy(phases = phasedPlan.phases.map(chunkBySize))
  }

  def chunkBySize(planPhase: PlanPhase): PlanPhase = {
    planPhase.copy(plan = chunkBySize(planPhase.plan))
  }

  def chunkBySize(plan: StepPlan): List[StepPlan] = {
    plan.getStepsWithPlanningInfo
      .grouped(threshold)
      .zipWithIndex
      .map(createStepPlanForSizedChunks(_, plan)).toList
  }

  def createStepPlanForSizedChunks(b: (mutable.Buffer[StepWithPlanningInfo], Int), plan: StepPlan): StepPlan = {
    def label(i: Int, p: StepPlan): String = s"Step ${1 + threshold * i} to ${Math.min(threshold * (i + 1), p.getStepsWithPlanningInfo.size())}"
    b match {
      case (ss, i) => plan.copy(label(i, plan), ss, findCheckpoints(ss, plan.getCheckpoints.toList))
    }
  }

}
