package com.xebialabs.deployit.deployment.planner

import com.xebialabs.deployit.deployment.planner.PlanSugar._
import com.xebialabs.deployit.plugin.satellite._
import com.xebialabs.xlplatform.satellite.Satellite

import scala.collection.convert.wrapAll._

trait Satellites {
  private[this] implicit lazy val actorSystem = SatelliteCommunicatorSystem.actorSystem

  private[this] def pingStepDescription(satellites: Set[Satellite]) = satelliteDescription("Check connection to", satellites)

  private[this] def preparingPhaseDescription(satellites: Set[Satellite]) = satelliteDescription("Preparing task on", satellites)

  private[this] def preparingPlanDescription(satellite: Satellite) = s"Prepare task for execution on ${satellite.getName}"

  private[this] def cleanUpPhaseDescription(satellites: Set[Satellite]) = satelliteDescription("Cleaning up", satellites)

  private[this] def satelliteDescription(verb: String, satellites: Set[Satellite]) = if (satellites.isEmpty) "" else {
    val s = if (satellites.size > 1) "s" else ""
    s"$verb satellite$s ${satelliteNames(satellites)}"
  }

  private[this] def satelliteNames(satellites: Set[Satellite]): String = {
    satellites.map(_.getName).toList.sorted.mkString(", ")
  }

  private[this] def extractSatellitesFromPlans(promotedPlan: PhasedPlan): Set[Satellite] =
    promotedPlan.phases.flatMap(phase => extractSatellites(phase.plan)).toSet

  private[this] def extractSatellites(plan: ExecutablePlan): Set[Satellite] = plan match {
    case executablePlan: ExecutablePlan if executablePlan.satellite != null => Set(executablePlan.satellite)
    case stepPlan: StepPlan => Set.empty
    case composite: CompositePlan => composite.getSubPlans.flatMap(extractSatellites).toSet
  }

  private[this] def addPingSteps(promotedPlan: PhasedPlan): PhasedPlan = {
    promotedPlan.copy(phases = promotedPlan.phases.map(phase => addPingSteps(phase, extractSatellites(phase.plan))).toList)
  }

  private def addPingSteps(phase: PlanPhase, allSatellites: Set[Satellite]): PlanPhase = {
    val pingSteps = allSatellites.map(PingSatelliteStep(_))
    val pingStepPlan = new StepPlan(pingStepDescription(allSatellites), pingSteps, phase.getListeners)
    phase.copy(plan = new SerialPlan(
      phase.plan.getDescription,
      List(pingStepPlan, phase.plan),
      phase.getListeners))
  }

  private[this] def addPreparePhase(promotedPlan: PhasedPlan, allSatellites: Set[Satellite], cleanupCache: TaskOnSatelliteCleanupCache): PhasedPlan = {
    implicit val uploader = SatelliteCommunicatorSystem.uploader
    val prepareSatellitePlans = allSatellites.toList.sortBy(_.getName).map { sat =>
      val checkStep = CheckExtensionsStep(sat)
      val sendToSatelliteStep = SendToSatelliteStep(sat, cleanupCache)
      new StepPlan(preparingPlanDescription(sat), List(checkStep, sendToSatelliteStep), promotedPlan.getListeners)
    }

    val preparingPlan = if (allSatellites.size == 1) prepareSatellitePlans.head else new ParallelPlan(
      preparingPhaseDescription(allSatellites),
      prepareSatellitePlans,
      promotedPlan.getListeners)

    val prepareSatellitePhase = new PlanPhase(preparingPlan, preparingPhaseDescription(allSatellites), promotedPlan.getListeners)
    promotedPlan.copy(phases = prepareSatellitePhase :: promotedPlan.phases.toList)
  }

  private[this] def addCleanUpSatellitePhase(promotedPlan: PhasedPlan, allSatellites: Set[Satellite], cleanupCache: TaskOnSatelliteCleanupCache): PhasedPlan = {
    val steps = allSatellites.map(CleanUpSatelliteStep(_, cleanupCache))
    val phaseDescription: String = cleanUpPhaseDescription(allSatellites)
    val cleanupPlan = new StepPlan(phaseDescription, steps, promotedPlan.getListeners)
    val cleanUpSatellitePhase = new PlanPhase(cleanupPlan, phaseDescription, promotedPlan.getListeners, true)
    promotedPlan.copy(phases = promotedPlan.phases.toList :+ cleanUpSatellitePhase)
  }

  def prepareForSatelliteExecution(plan: PhasedPlan): PhasedPlan = {
    val satellites = extractSatellitesFromPlans(plan)
    if (satellites.nonEmpty) {
      val cleanupCache = new TaskOnSatelliteCleanupCache
      addCleanUpSatellitePhase(addPreparePhase(addPingSteps(plan), satellites, cleanupCache), satellites, cleanupCache)
    } else {
      plan
    }
  }

}

object Satellites extends Satellites
