/**
 * Copyright 2014-2018 XebiaLabs Inc. and its affiliates. Use is subject to terms of the enclosed Legal Notice.
 */
package com.xebialabs.deployit.booter.remote.service;

import java.io.*;
import java.net.URLEncoder;
import java.util.List;

import com.xebialabs.deployit.engine.api.dto.ImportFromUrlParams;
import org.apache.http.Header;
import org.apache.http.HttpEntity;
import org.apache.http.HttpResponse;
import org.apache.http.client.methods.HttpPost;
import org.apache.http.entity.mime.MultipartEntityBuilder;
import org.apache.http.entity.mime.content.FileBody;
import org.apache.http.util.EntityUtils;

import com.thoughtworks.xstream.XStream;
import com.thoughtworks.xstream.converters.DataHolder;
import com.thoughtworks.xstream.core.MapBackedDataHolder;
import com.thoughtworks.xstream.io.HierarchicalStreamReader;

import com.xebialabs.deployit.booter.remote.BooterConfig;
import com.xebialabs.deployit.booter.remote.DeployitCommunicator;
import com.xebialabs.deployit.booter.remote.HttpClientHolder;
import com.xebialabs.deployit.booter.remote.Proxies;
import com.xebialabs.deployit.booter.remote.resteasy.DeployitClientException;
import com.xebialabs.deployit.engine.api.PackageService;
import com.xebialabs.deployit.engine.api.dto.FileUpload;
import com.xebialabs.xltype.serialization.xstream.XStreamReaderWriter;
import com.xebialabs.deployit.plugin.api.udm.ConfigurationItem;

import static java.lang.String.format;


/**
 * Imports a package bypassing resteasy. It's needed because of issue described at DEPLOYITPB-3999.
 */
public class StreamingImportingService implements PackageService {

    private HttpClientHolder httpClientHolder;
    private BooterConfig booterConfig;

    private Proxies proxies;

    /**
     * Use the second constructor
     */
    @Deprecated
    public StreamingImportingService(DeployitCommunicator communicator, Proxies proxies) {
        this.httpClientHolder = communicator.getHttpClientHolder();
        this.booterConfig = communicator.getConfig();
        this.proxies = proxies;
    }

    public StreamingImportingService(HttpClientHolder httpClientHolder, BooterConfig config, Proxies proxies) {
        this.httpClientHolder = httpClientHolder;
        this.booterConfig = config;
        this.proxies = proxies;
    }

    @Override
    public List<String> list() {
        return getPackageProxy().list();
    }

    private PackageService getPackageProxy() {
        return proxies.getProxyInstance(PackageService.class);
    }

    @Override
    public ConfigurationItem importPackage(final String file) {
        return getPackageProxy().importPackage(file);
    }

    @SuppressWarnings("deprecation")
    @Override
    public ConfigurationItem fetch(String url) {
        return getPackageProxy().fetch(new ImportFromUrlParams(url));
    }

    @Override
    public ConfigurationItem fetch(ImportFromUrlParams params) {
        return getPackageProxy().fetch(params);
    }

    /**
     * This method should accept LocalFileUpload instance as second parameter.
     */
    @Override
    public ConfigurationItem upload(final String file, final FileUpload form) {
        if (!(form instanceof LocalFileUpload)) {
            throw new IllegalArgumentException();
        }

        File fileToUpload = ((LocalFileUpload) form).getFile();

        try {
            return responseToCi(httpClientHolder.execute(httpPost(fileToUpload)));
        } catch (IOException e) {
            throw new ImportException("Failed to import dar file", e);
        }
    }


    // Private helpers

    private HttpPost httpPost(final File file) throws UnsupportedEncodingException {
        BooterConfig config = booterConfig;

        HttpEntity entity = MultipartEntityBuilder.create().addPart("fileData", new FileBody(file)).build();

        HttpPost post = new HttpPost(config.getUrl() + "/package/upload/" + URLEncoder.encode(file.getName(), "UTF-8"));
        post.setEntity(entity);
        return post;
    }

    private ConfigurationItem responseToCi(HttpResponse response) {
        try {
            final int statusCode = response.getStatusLine().getStatusCode();
            if (!isOk(statusCode)) {
                tryDeployitException(response);
                tryUnhandledException(response);
                throw new RuntimeException("Failed : HTTP error code : " + statusCode);
            }

            XStream xStream = XStreamReaderWriter.getConfiguredXStream();
            HierarchicalStreamReader reader = XStreamReaderWriter.HIERARCHICAL_STREAM_DRIVER.createReader(response.getEntity().getContent());
            final DataHolder dataHolder = new MapBackedDataHolder();
            dataHolder.put("BOOTER_CONFIG", booterConfig.getKey());
            return (ConfigurationItem) xStream.unmarshal(reader, null, dataHolder);
        } catch (IOException e) {
            throw new ImportException("Failed to import dar file", e);
        }  finally {
            if (response != null) {
                try {
                    EntityUtils.consume(response.getEntity());
                } catch (IOException e) {
                    throw new RuntimeException(e);
                }
            }
        }
    }

    private static boolean isOk(final int statusCode) {
        return statusCode < 300;
    }

    private static void tryDeployitException(HttpResponse response) throws IOException {
        if (response.containsHeader("X-Deployit-Exception")) {
            final Header header = response.getHeaders("X-Exception-Type")[0];
            final String value = header.getValue();
            String message = EntityUtils.toString(response.getEntity());
            throw new DeployitClientException(value + ":" + message, response.getStatusLine().getStatusCode());
        }
    }

    private static void tryUnhandledException(HttpResponse response) throws IOException {
        if (response.containsHeader("Unhandled-Exception")) {
            final Header header = response.getHeaders("Exception-Type")[0];
            final String exceptionType = header.getValue();
            String unhandledMessage = EntityUtils.toString(response.getEntity());
            throw new RuntimeException(format("%s: %s", exceptionType, unhandledMessage));
        }
    }

    public static class LocalFileUpload extends FileUpload {

        private File file;

        public LocalFileUpload(File file) throws FileNotFoundException {
            this.file = file;
        }

        public File getFile() {
            return file;
        }

        @Override
        public InputStream getFileData() {
            try {
                return new FileInputStream(file);
            } catch (FileNotFoundException e) {
                throw new ImportException("Can not open file", e);
            }
        }
    }

    public static class ImportException extends RuntimeException {
        public ImportException(final String message, final Throwable cause) {
            super(message, cause);
        }
    }
}
