/**
 * Copyright 2014-2018 XebiaLabs Inc. and its affiliates. Use is subject to terms of the enclosed Legal Notice.
 */
package com.xebialabs.deployit.booter.remote;

import java.io.Serializable;
import java.util.ArrayList;
import java.util.List;

import com.xebialabs.deployit.plugin.api.reflect.DescriptorRegistryId;
import com.xebialabs.xltype.serialization.xstream.XStreamReaderWriter;

import org.apache.http.HttpRequestInterceptor;

import static java.lang.String.format;

public class BooterConfig implements Serializable, DescriptorRegistryId {

    public enum Protocol {
        HTTP("http"), HTTPS("https");

        private String protocol;

        Protocol(String protocol) {
            this.protocol = protocol;
        }

        public String getProtocol() {
            return protocol;
        }

    }

    private Protocol protocol = Protocol.HTTP;
    private String host;
    private int port = -1;
    private String proxyHost;
    private int proxyPort = -1;
    private String context = "deployit";
    private String contextRoot = "";
    private String username;
    private String password;
    private int connectionPoolSize;
    private int socketTimeout = 1000;
    private List<HttpRequestInterceptor> httpRequestInterceptors = new ArrayList<>();
    private XStreamReaderWriter xStreamReaderWriter;
    private boolean withoutCredentials = false;

    public static Builder builder() {
        return new Builder();
    }

    public static Builder builder(BooterConfig config) {
        return new Builder(config);
    }

    public static class Builder {
        private BooterConfig config;

        public Builder() {
            this.config = new BooterConfig();
        }

        public Builder(final BooterConfig config) {
            this.config = config;
        }

        public Builder withProtocol(Protocol protocol) {
            config.protocol = protocol;
            return this;
        }

        public Builder withHost(String host) {
            config.host = host;
            return this;
        }

        public Builder withPort(int port) {
            config.port = port;
            return this;
        }

        public Builder withProxyHost(String proxyHost) {
            config.proxyHost = proxyHost;
            return this;
        }

        public Builder withProxyPort(int proxyPort) {
            config.proxyPort = proxyPort;
            return this;
        }

        public Builder withContext(String context) {
            config.context = context;
            if (!config.context.endsWith("deployit")) {
                config.context = config.context + "/deployit";
            }
            if (config.context.startsWith("/")) {
                config.context = config.context.substring(1);
            }
            int lastSlashIndex = config.context.lastIndexOf("/");
            if (lastSlashIndex != -1) {
                config.contextRoot = config.context.substring(0, lastSlashIndex);
            }
            return this;
        }

        public Builder withConnectionPoolSize(int poolSize) {
            config.connectionPoolSize = poolSize;
            return this;
        }

        public Builder withSocketTimeout(int socketTimeout) {
            config.socketTimeout = socketTimeout;
            return this;
        }

        public Builder withHttpRequestInterceptor(HttpRequestInterceptor interceptor) {
            config.httpRequestInterceptors.add(interceptor);
            return this;
        }

        public Builder withHttpRequestInterceptors(List<HttpRequestInterceptor> interceptors) {
            config.httpRequestInterceptors = interceptors;
            return this;
        }

        public Builder withCredentials(String username, String password) {
            config.username = username;
            config.password = password;
            return this;
        }

        public Builder withXStreamReaderWriter(XStreamReaderWriter xStreamReaderWriter) {
            config.xStreamReaderWriter = xStreamReaderWriter;
            return this;
        }

        public Builder withoutCredentials() {
            config.withoutCredentials = true;
            return this;
        }

        public BooterConfig build() {
            return config;
        }
    }

    public String getHost() {
        return host;
    }

    public int getPort() {
        return port == -1 ? (protocol == Protocol.HTTPS ? 4517 : 4516) : port;
    }

    public String getProxyHost() {
        return proxyHost;
    }

    public int getProxyPort() {
        return proxyPort;
    }

    public Protocol getProtocol() {
        return protocol;
    }

    public String getUsername() {
        return username;
    }

    public String getPassword() {
        return password;
    }

    public int getConnectionPoolSize() {
        return connectionPoolSize;
    }

    public int getSocketTimeout() {
        return socketTimeout;
    }

    public List<HttpRequestInterceptor> getHttpRequestInterceptors() {
        return httpRequestInterceptors;
    }

    public String getContext() {
        return context;
    }

    public String getUrl() {
        return format("%s://%s:%d/%s", protocol.getProtocol(), host, getPort(), getContext());
    }

    public String getExtensionApiUrl() {
        String context = contextRoot;
        if (context.length() > 0) {
            context = context + "/";
        }
        return format("%s://%s:%d/%s", protocol.getProtocol(), host, getPort(), context + "api");
    }

    public String getKey() {
        String key = format("%s://%s:%s@%s:%d/%s", protocol.getProtocol(), username, password, host, getPort(), getContext());
        if (isProxied()) {
            key = format("%s via proxy %s:%s", key, proxyHost, proxyPort);
        }
        return key;
    }

    public XStreamReaderWriter getXStreamReaderWriter() {
        return xStreamReaderWriter;
    }

    public boolean isWithoutCredentials() {
        return withoutCredentials;
    }

    public boolean isSecure() {
        return protocol == Protocol.HTTPS;
    }

    public boolean isProxied() {
        return proxyPort != -1 && proxyHost != null;
    }

    @Override
    public boolean equals(Object o) {
        if (this == o) return true;
        if (o == null || getClass() != o.getClass()) return false;

        BooterConfig that = (BooterConfig) o;

        if (!host.equals(that.host)) return false;
        if (port != that.port) return false;
        if (!password.equals(that.password)) return false;
        if ((proxyHost == null && that.proxyHost != null) || (proxyHost != null && !proxyHost.equals(that.proxyHost)))
            return false;
        if (proxyPort != that.proxyPort) return false;
        if (protocol != that.protocol) return false;
        if (!username.equals(that.username)) return false;
        if (!getContext().equals(that.getContext())) return false;

        return true;
    }

    @Override
    public int hashCode() {
        int result = firstNonNull(protocol, "").hashCode();
        result = 31 * result + firstNonNull(host, "").hashCode();
        result = 31 * result + port;
        if (proxyHost != null) result = 31 * result + firstNonNull(proxyHost, "").hashCode();
        result = 31 * result + proxyPort;
        result = 31 * result + firstNonNull(username, "").hashCode();
        result = 31 * result + firstNonNull(password, "").hashCode();
        result = 31 * result + getContext().hashCode();
        return result;
    }

    private <T> T firstNonNull(T first, T second) {
        if (first != null) {
            return first;
        } else if (second != null) {
            return second;
        } else {
            throw new NullPointerException();
        }
    }
}
