package com.xebialabs.deployit.booter.remote;

import java.io.Serializable;

import static java.lang.String.format;

public class BooterConfig implements Serializable {

    public enum Protocol {
        HTTP("http"), HTTPS("https");

        private String protocol;

        Protocol(String protocol) {
            this.protocol = protocol;
        }

        public String getProtocol() {
            return protocol;
        }

    }

    private Protocol protocol = Protocol.HTTP;
    private String host;
    private int port = -1;
    private String context = "deployit";
    private String username;
    private String password;

    public static Builder builder() {
        return new Builder();
    }

    public static class Builder {
        private BooterConfig config = new BooterConfig();

        public Builder withProtocol(Protocol protocol) {
            config.protocol = protocol;
            return this;
        }

        public Builder withHost(String host) {
            config.host = host;
            return this;
        }

        public Builder withPort(int port) {
            config.port = port;
            return this;
        }

        public Builder withContext(String context) {
            config.context = context;
            return this;
        }

        public Builder withCredentials(String username, String password) {
            config.username = username;
            config.password = password;
            return this;
        }

        public BooterConfig build() {
            return config;
        }
    }

    public String getHost() {
        return host;
    }

    public int getPort() {
        return port == -1 ? (protocol == Protocol.HTTPS ? 4517 : 4516) : port;
    }

    public Protocol getProtocol() {
        return protocol;
    }

    public String getUsername() {
        return username;
    }

    public String getPassword() {
        return password;
    }

    public String getContext() {
        if (!context.endsWith("deployit")) {
            context = context + "/deployit";
        }
        if (context.startsWith("/")) {
            context = context.substring(1);
        }
        return context;
    }

    public String getUrl() {
        return format("%s://%s:%d/%s", protocol.getProtocol(), host, getPort(), getContext());
    }

    public boolean isSecure() {
        return protocol == Protocol.HTTPS;
    }

    @Override
    public boolean equals(Object o) {
        if (this == o) return true;
        if (o == null || getClass() != o.getClass()) return false;

        BooterConfig that = (BooterConfig) o;

        if (!host.equals(that.host)) return false;
        if (!password.equals(that.password)) return false;
        if (protocol != that.protocol) return false;
        if (!username.equals(that.username)) return false;

        return true;
    }

    @Override
    public int hashCode() {
        int result = protocol.hashCode();
        result = 31 * result + host.hashCode();
        result = 31 * result + username.hashCode();
        result = 31 * result + password.hashCode();
        return result;
    }
}
