package com.xebialabs.deployit.engine.tasker;

import com.xebialabs.deployit.engine.spi.execution.ExecutionStateListener;
import com.xebialabs.deployit.engine.spi.execution.StepExecutionStateEvent;
import com.xebialabs.deployit.engine.spi.execution.TaskExecutionStateEvent;
import com.xebialabs.deployit.plugin.api.inspection.InspectionContext;
import nl.javadude.t2bus.EventHandlerStrategy;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;

import java.io.Serializable;
import java.util.List;
import java.util.Map;

import static com.google.common.base.Preconditions.checkState;
import static com.google.common.collect.Maps.newHashMap;
import static com.xebialabs.deployit.engine.tasker.EventBusAdapter.tryRegister;
import static com.xebialabs.deployit.engine.tasker.EventBusAdapter.tryUnregister;

public final class TaskExecutionContext implements Serializable {
    private transient EventBusAdapter bus;
    private Map<String, Object> attributes = newHashMap();
    private TaskInspectionContext inspectionContext;

    TaskExecutionContext(List<ExecutionStateListener> listeners, TaskInspectionContext context) {
        for (ExecutionStateListener listener : listeners) {
            attributes.put(listener.getClass().getName(), listener);
        }

        this.inspectionContext = context;
    }

    public Object getAttribute(String name) {
        return attributes.get(name);
    }

    public void setAttribute(String name, Object value) {
        tryRegister(value, bus);
        attributes.put(name, value);
    }

    public void unsetAttribute(String name) {
        tryUnregister(attributes.get(name), bus);
        attributes.remove(name);
    }

    public InspectionContext getInspectionContext() {
        checkState(inspectionContext != null, "InspectionContext is not set, are you currently running an Inspection?");
        return inspectionContext;
    }

    void initWithBus(EventBusAdapter bus) {
        this.bus = bus;
        // hook up pre-existing triggers.
        for (Object o : attributes.values()) {
            tryRegister(o, this.bus);
        }

    }

    void publish(TaskExecutionStateEvent e, EventHandlerStrategy handler) {
        checkState(bus != null, "No EventBus is set on the context");
        bus.post(e, handler);
    }

    void publish(StepExecutionStateEvent e, final EventHandlerStrategy handler) {
        checkState(bus != null, "No EventBus is set on the context");
        bus.post(e, handler);
    }

    void publish(TaskExecutionStateEvent e) {
        checkState(bus != null, "No EventBus is set on the context");
        bus.post(e);
    }

    void publish(StepExecutionStateEvent e) {
        checkState(bus != null, "No EventBus is set on the context");
        bus.post(e);
    }

    Map<String, Object> getAttributes() {
        return attributes;
    }

    private static final Logger logger = LoggerFactory.getLogger(TaskExecutionContext.class);
}
