package com.xebialabs.deployit.test.support;

import java.io.File;
import java.io.IOException;
import java.net.URISyntaxException;
import java.net.URL;
import java.nio.charset.Charset;
import java.util.ArrayList;
import java.util.Arrays;
import java.util.List;
import java.util.Map;
import java.util.Set;

import org.hamcrest.CoreMatchers;
import org.jdom2.Document;
import org.jdom2.Element;
import org.jdom2.JDOMException;
import org.jdom2.input.SAXBuilder;
import org.jdom2.input.sax.XMLReaders;
import org.junit.After;
import org.junit.AfterClass;
import org.junit.Assert;
import org.junit.Before;
import org.junit.BeforeClass;
import org.junit.Rule;
import org.junit.Test;
import org.junit.rules.TemporaryFolder;
import org.junit.runner.RunWith;

import com.google.common.base.Function;
import com.google.common.base.Preconditions;
import com.google.common.base.Predicate;
import com.google.common.collect.Iterables;
import com.google.common.collect.Lists;
import com.google.common.collect.Sets;
import com.google.common.io.Files;

import com.xebialabs.deployit.booter.local.LocalBooter;
import com.xebialabs.deployit.booter.local.utils.Strings;
import com.xebialabs.deployit.deployment.planner.DeltaSpecificationBuilder;
import com.xebialabs.deployit.plugin.api.deployment.specification.DeltaSpecification;
import com.xebialabs.deployit.plugin.api.flow.StepExitCode;
import com.xebialabs.deployit.plugin.api.reflect.Descriptor;
import com.xebialabs.deployit.plugin.api.reflect.DescriptorRegistry;
import com.xebialabs.deployit.plugin.api.reflect.PropertyDescriptor;
import com.xebialabs.deployit.plugin.api.reflect.Type;
import com.xebialabs.deployit.plugin.api.udm.ConfigurationItem;
import com.xebialabs.deployit.plugin.api.udm.Container;
import com.xebialabs.deployit.plugin.api.udm.Deployed;
import com.xebialabs.deployit.plugin.api.udm.artifact.Artifact;
import com.xebialabs.deployit.plugin.api.validation.ValidationMessage;
import com.xebialabs.deployit.test.deployment.DeployitTester;
import com.xebialabs.deployit.test.support.junit.Parameterized;
import com.xebialabs.deployit.test.support.junit.Parameterized.Parameters;
import com.xebialabs.overthere.local.LocalFile;

import static com.google.common.base.Joiner.on;
import static com.google.common.base.Preconditions.checkArgument;
import static com.google.common.base.Preconditions.checkNotNull;
import static com.google.common.collect.Iterables.filter;
import static com.google.common.collect.Lists.newArrayList;
import static com.google.common.collect.Maps.newHashMap;
import static com.google.common.collect.Sets.newHashSet;
import static com.google.common.io.Files.copy;
import static com.google.common.io.Resources.newInputStreamSupplier;
import static com.xebialabs.deployit.plugin.api.reflect.PropertyKind.LIST_OF_CI;
import static com.xebialabs.deployit.test.support.TestUtils.createDeployedApplication;
import static com.xebialabs.deployit.test.support.TestUtils.createDeploymentPackage;
import static com.xebialabs.deployit.test.support.TestUtils.createEnvironment;
import static com.xebialabs.overthere.util.OverthereUtils.getName;
import static java.lang.String.format;
import static org.junit.Assert.assertThat;
import static org.junit.Assert.fail;
import static org.junit.Assume.assumeTrue;

/**
 * @see <a
 *      href="https://intranet.xebia.com/confluence/display/Labs/Deployed+Itesting+with+xml">https://intranet.xebia.com/confluence/display/Labs/Deployed Itesting with xml</a>
 */
@SuppressWarnings("rawtypes")
@RunWith(Parameterized.class)
public abstract class DeployedItestBase {
    static {
        LocalBooter.bootWithoutGlobalContext();
    }

    @Rule
    public TemporaryFolder folder = new TemporaryFolder();

    private static final String DEPLOYED_ITEST_DIR = "deployed-test-scripts";

    private File deployedPropertiesFile;

    protected ItestTopology topology;

    protected Container container;
    private static DeployitTester tester;

    private List<Type> additionalTypesToDiscover = newArrayList();
    private Map<String, Deployed> deployedDependencies = newHashMap();
    private Set<PropertyDescriptor> requiredForInspection = newHashSet();

    protected enum DeployMode {
        CREATE,
        UPDATE_ONLY
    }

    private DeployMode deployMode = DeployMode.CREATE;

    private static TestExecutionContext context;

    public DeployedItestBase(String description, File deployedPropertiesFile, ItestTopology topology, Container container) {
        this.deployedPropertiesFile = deployedPropertiesFile;
        this.topology = topology;
        this.container = container;
    }

    @Parameters(name = "{0}")
    public static List<Object[]> getTargets() throws URISyntaxException, IOException, JDOMException {
        List<Object[]> constructorArgsList = new ArrayList<Object[]>();
        Map<String, ItestTopology> topologies = ItestTopology.load();

        for (File deployedItestFile : getDeployedItestFiles()) {
            String desc = "Test " + deployedItestFile.getName() + " on ";
            for (ItestTopology topology : topologies.values()) {
                if (ItestTopology.isItestEnabled(topology.getId(), topology.isEnabledByDefault())) {
                    for (Container container : topology.getTargets()) {
                        constructorArgsList.add(new Object[] { desc + container, deployedItestFile, topology, container });
                    }
                }
            }
        }
        return constructorArgsList;
    }

    @Before
    public void clearAdditionalTypesToDiscoverAndRequiredDeployeds() {
        deployedDependencies.clear();
        additionalTypesToDiscover.clear();
    }

    @After
    public void clearInspectionContext() {
        context.clearInspectionContext();
    }

    @BeforeClass
    public static void setup() {
        tester = new DeployitTester();
        context = new TestExecutionContext(DeployedItestBase.class);
    }

    @AfterClass
    public static void tearDown() {
        context.destroy();
    }

    private static Iterable<File> getDeployedItestFiles() throws URISyntaxException, IOException {
        File deployedItestDir = getDeployedItestDir();
        final List<String> enabledItests = getEnabledItests();
        return Iterables.filter(Arrays.asList(deployedItestDir.listFiles()), new Predicate<File>() {
            @Override
            public boolean apply(File input) {
                String name = input.getName();
                return (name.endsWith(".xml")) &&
                    (enabledItests.isEmpty() || enabledItests.contains(name));
            }
        });
    }

    private static File getDeployedItestDir() throws URISyntaxException {
        URL deployedItestDirUrl = Thread.currentThread().getContextClassLoader().getResource(DEPLOYED_ITEST_DIR);
        checkNotNull(deployedItestDirUrl);
        return new File(deployedItestDirUrl.toURI());
    }

    private static List<String> getEnabledItests() throws URISyntaxException, IOException {
        File deployedItestDir = getDeployedItestDir();
        File enabledItestsFile = new File(deployedItestDir, "enabled-itests");
        List<String> enabledItests = newArrayList();
        if (enabledItestsFile.exists()) {
            List<String> tests = Files.readLines(enabledItestsFile, Charset.defaultCharset());
            enabledItests.addAll(tests);
        }
        return enabledItests;
    }

    @Test
    public void shouldCreateAndDestroyDeployed() throws Exception {
        List<Deployed> deployeds = readDeployeds();

        assumeTrue(deployeds.size() > 0);

        List<Deployed> deployedsWithDependencies = newArrayList(deployedDependencies.values());
        deployedsWithDependencies.addAll(deployeds);

        assignDeployedsToContainer(deployedsWithDependencies);

        if (shouldCheckDeployedsDoNotExistBeforeCreating()) {
            assertDeployedsDoNotExist(deployedsWithDependencies);
            context.clearInspectionContext();
        }

        if (shouldCreateDeployed()) {
            createDeployeds(deployedsWithDependencies);
            context.clearInspectionContext();
        }

        if (shouldCheckDeployedsCreatedCorrectly()) {
            assertDeployedsWereCreatedCorrectly(deployedsWithDependencies);
            context.clearInspectionContext();
        }

        if (shouldDestroyDeployeds()) {
            destroyDeployeds(deployedsWithDependencies);
            context.clearInspectionContext();
        }

        if (shouldCheckDeployedsDoNotExistAfterDelete()) {
            assertDeployedsDoNotExist(deployedsWithDependencies);
            context.clearInspectionContext();
        }
    }

    protected void setDeployMode(String deployModeAttribute) {
        if (deployModeAttribute == null) {
            return;
        }
        try {
            deployMode = DeployMode.valueOf(deployModeAttribute.toUpperCase());
        } catch (IllegalArgumentException e) {
            throw new IllegalArgumentException(String.format("Unsupported deployMode value '%s' on itest tag expected [%s]",
                deployModeAttribute, on("|").join(newArrayList(DeployMode.values()))));
        }
    }

    protected DeployMode getDeployMode() {
        return deployMode;
    }

    private List<Deployed> readDeployeds() throws JDOMException, IOException {
        SAXBuilder sb = new SAXBuilder(XMLReaders.NONVALIDATING);
        Document deployedProperties = sb.build(deployedPropertiesFile);

        Map<String, Deployed> deployeds = newHashMap();
        Element rootElement = deployedProperties.getRootElement();
        List<Element> deployedGroupElements = rootElement.getChildren("deployeds");

        setDeployMode(rootElement.getAttributeValue("deployMode"));

        for (Element deployedGroupElement : deployedGroupElements) {
            String additionalTypes = deployedGroupElement.getAttributeValue("additionalTypesToDiscover");
            if (Strings.isNotBlank(additionalTypes)) {
                for (String t : additionalTypes.split(",")) {
                    additionalTypesToDiscover.add(Type.valueOf(t.trim()));
                }
            }
            String target = deployedGroupElement.getAttributeValue("target");
            String alwaysDeployAttrVal = deployedGroupElement.getAttributeValue("alwaysDeployToTarget");
            boolean alwaysDeployToTarget = alwaysDeployAttrVal != null && alwaysDeployAttrVal.equals("true");
            List<Element> deployedElements = deployedGroupElement.getChildren();
            for (Element each : deployedElements) {
                convertDeployedElementToDeployed(each, target, alwaysDeployToTarget, deployeds);
            }
        }

        return newArrayList(deployeds.values());
    }

    @SuppressWarnings("unchecked")
    private void convertDeployedElementToDeployed(Element deployedElement, String target, boolean alwaysDeployToTarget, Map<String, Deployed> deployeds)
        throws IOException {
        Descriptor d = DescriptorRegistry.getDescriptor(deployedElement.getName());
        checkArgument(d.isAssignableTo(Deployed.class), "Type [%s] is not assignable to udm.Deployed", d.getType());

        Deployed item = d.newInstance();

        if (item instanceof Artifact) {
            Element fileNameElm = deployedElement.getChild("fileName");
            checkArgument(fileNameElm != null, "Type [%s] is an artifact and must specify fileName that can be resolved from the classpath", d.getType());
            File artifact = folder.newFile(getName(fileNameElm.getValue()));
            URL aURL = getClass().getClassLoader().getResource(fileNameElm.getValue());
            copy(newInputStreamSupplier(aURL), artifact);
            ((Artifact) item).setFile(LocalFile.valueOf(artifact));
        }

        item.setId(Preconditions.checkNotNull(deployedElement.getAttributeValue("id"), "Element [%s] has no id attribute", deployedElement.getName()));
        for (Element p : deployedElement.getChildren()) {
            if (p.getName().equals("fileName")) {
                continue;
            }

            PropertyDescriptor pd = Preconditions.checkNotNull(d.getPropertyDescriptor(p.getName()), "Property [%s.%s] does not exist", d.getType(),
                p.getName());
            if (p.getAttributeValue("inspectionProperty") != null && p.getAttributeValue("inspectionProperty").equals("true")) {
                requiredForInspection.add(pd);
            }
            switch (pd.getKind()) {
            case CI:
                pd.set(item, resolveCiReference(p, pd, item, deployeds));
                break;
            case MAP_STRING_STRING:
                Map<String, String> map = newHashMap();
                for (Element e : p.getChildren("entry")) {
                    map.put(Preconditions.checkNotNull(e.getAttributeValue("key")), Preconditions.checkNotNull(e.getValue()));
                }
                pd.set(item, map);
                break;
            case SET_OF_STRING:
                Set<String> setOfString = newHashSet();
                for (Element v : p.getChildren("value")) {
                    setOfString.add(v.getValue());
                }
                pd.set(item, setOfString);
                break;
            case SET_OF_CI:
                Set<ConfigurationItem> setOfCi = newHashSet();
                for (Element v : p.getChildren("value")) {
                    setOfCi.add(resolveCiReference(v, pd, item, deployeds));
                }
                pd.set(item, setOfCi);
                break;
            default:
                pd.set(item, p.getValue());
            }
        }

        if (target != null) {
            Type containerScopeType = Type.valueOf(target);
            if (container.getType().instanceOf(containerScopeType)) {
                deployeds.put(item.getId(), item);
            } else if (alwaysDeployToTarget) {
                item.setContainer(topology.findFirstMatchingTarget(containerScopeType));
                deployedDependencies.put(item.getId(), item);
            }
        } else {
            deployeds.put(item.getId(), item);
        }
    }

    private ConfigurationItem resolveCiReference(Element p, PropertyDescriptor pd, Deployed item, Map<String, Deployed> deployeds) {
        String ciRefId = p.getValue().trim();
        if (deployeds.containsKey(ciRefId)) {
            return deployeds.get(ciRefId);
        }

        if (deployedDependencies.containsKey(ciRefId)) {
            return deployedDependencies.get(ciRefId);
        }

        ciRefId = topology.replacePlaceholders(ciRefId);
        ConfigurationItem ciRef = topology.getItems().get(ciRefId);
        if (ciRef == null) {
            ciRef = DescriptorRegistry.getDescriptor(pd.getReferencedType()).newInstance();
            ciRef.setId(ciRefId);
        }

        return ciRef;
    }

    private void assignDeployedsToContainer(List<Deployed> deployeds) {
        for (Deployed deployed : deployeds) {
            if (deployed.getContainer() == null) {
                deployed.setContainer(container);
            }
            deployed.setId(deployed.getContainer().getId() + "/" + deployed.getId());
        }
    }

    private void createDeployeds(List<Deployed> deployeds) {
        DeltaSpecificationBuilder builder = new DeltaSpecificationBuilder().initial(createDeployedApplication(createDeploymentPackage("1.0"),
            createEnvironment(container)));
        for (Deployed d : deployeds) {
            List<ValidationMessage> validationErrors = DescriptorRegistry.getDescriptor(d.getType()).validate(d);
            if (validationErrors.isEmpty()) {
                builder.create(d);
            } else {
                fail("Deployed has validation errors: " + validationErrors);
            }
        }
        DeltaSpecification spec = builder.build();
        List<com.xebialabs.deployit.plugin.api.flow.Step> resolvedPlan = tester.resolvePlan(spec);
        StepExitCode result = tester.executePlan(resolvedPlan, context);
        assertThat(result, CoreMatchers.is(StepExitCode.SUCCESS));
    }

    private void destroyDeployeds(List<Deployed> deployeds) {
        DeltaSpecificationBuilder builder = new DeltaSpecificationBuilder().initial(createDeployedApplication(createDeploymentPackage("1.0"),
            createEnvironment(container)));
        for (Deployed d : deployeds) {
            builder.destroy(d);
        }
        DeltaSpecification spec = builder.build();

        List<com.xebialabs.deployit.plugin.api.flow.Step> resolvedPlan = tester.resolvePlan(spec);
        StepExitCode result = tester.executePlan(resolvedPlan, context);
        assertThat(result, CoreMatchers.is(StepExitCode.SUCCESS));
    }

    private void assertDeployedsDoNotExist(List<Deployed> deployeds) {
        try {
            List<Deployed> actualDeployeds = inspectDeployeds(deployeds);

            if (!actualDeployeds.isEmpty()) {
                fail(format("Expected deployeds [%s] to not exist", on(",").join(actualDeployeds)));
            }
        } catch (RuntimeException e) {
            if (e.getMessage().contains("Step failed:") || e.getMessage().contains("CliScriptException")) {
                // That's ok, we want it! Success!
            } else {
                throw e;
            }
        }
    }

    private void assertDeployedsWereCreatedCorrectly(List<Deployed> expectedDeployeds) {
        List<Deployed> actualDeployeds = inspectDeployeds(expectedDeployeds);

        if (actualDeployeds.size() != expectedDeployeds.size()) {
            fail(format("Expected deployeds [%s] to exist, but only found [%s]", on(",").join(expectedDeployeds), on(",").join(actualDeployeds)));
        }

        for (Deployed expected : expectedDeployeds) {
            int i = actualDeployeds.indexOf(expected);
            checkArgument(i != -1, "Cannot find [%s] in list of existing deployeds", expected);
            Deployed actual = actualDeployeds.get(i);
            assertDeployedWasCreatedCorrectly(expected, actual);
        }
    }

    @SuppressWarnings("unchecked")
    private static void assertDeployedWasCreatedCorrectly(Deployed expectedDeployed, Deployed actualDeployed) {
        Descriptor dd = DescriptorRegistry.getDescriptor(expectedDeployed.getType());
        for (PropertyDescriptor pd : dd.getPropertyDescriptors()) {
            if (pd.isHidden() || pd.isPassword() || pd.getKind() == LIST_OF_CI) {
                continue;
            }

            switch (pd.getKind()) {
            case MAP_STRING_STRING:
                Map<String, String> expectedMap = (Map<String, String>) pd.get(expectedDeployed);
                Map<String, String> inspectedMap = (Map<String, String>) pd.get(actualDeployed);
                if (expectedMap != null) {
                    Assert.assertEquals(inspectedMap.size(), expectedMap.size());
                    for (Map.Entry<String, String> eachEntry : inspectedMap.entrySet()) {
                        assertThat(inspectedMap.get(eachEntry.getKey()), CoreMatchers.equalTo(expectedMap.get(eachEntry.getKey())));
                    }
                }
                break;
            case SET_OF_STRING:
                Integer diffInStringSets = Sets.symmetricDifference((Set<String>) pd.get(expectedDeployed), (Set<String>) pd.get(actualDeployed)).size();
                assertThat("Expected " + pd.get(expectedDeployed) + " but was " + pd.get(actualDeployed), diffInStringSets, CoreMatchers.equalTo(0));
                break;
            case SET_OF_CI:
                Set<ConfigurationItem> set1 = newHashSet((Set<ConfigurationItem>) pd.get(expectedDeployed));
                Set<ConfigurationItem> set2 = newHashSet((Set<ConfigurationItem>) pd.get(actualDeployed));
                Integer diffInCiSets = Sets.symmetricDifference(set1, set2).size();
                assertThat("Expected " + pd.get(expectedDeployed) + " but was " + pd.get(actualDeployed), diffInCiSets, CoreMatchers.equalTo(0));
                break;
            default:
                Object inspectedValue = pd.get(actualDeployed);
                Object expectedValue = pd.get(expectedDeployed);
                if (expectedValue != null) {
                    assertThat(inspectedValue, CoreMatchers.equalTo(expectedValue));
                }
            }
        }
    }

    @SuppressWarnings("unchecked")
    private List<Deployed> inspectDeployeds(final List<Deployed> deployeds) {
        List<Type> typesToDiscover = newArrayList();
        typesToDiscover.addAll(additionalTypesToDiscover);
        typesToDiscover.addAll(Lists.transform(deployeds, new Function<Deployed, Type>() {
            public Type apply(Deployed input) {
                return input.getType();
            }
        }));

        List<ConfigurationItem> discovered = newArrayList();
        ConfigurationItem ciToPerformInspectionOn = getCiToPerformInspectionOn(typesToDiscover);

        if (ciToPerformInspectionOn != null) {
            // perform new style discovery
            discovered = tester.runInspectionTask(ciToPerformInspectionOn, context);
        } else {
            // perform old style discovery
            if (!additionalTypesToDiscover.isEmpty()) {
                throw new UnsupportedOperationException("The discovery of additional typess is not supported using old style discovery.");
            }
            for (Deployed<?, ?> deployed : cloneForInspection(deployeds)) {
                discovered.addAll(tester.runInspectionTask(deployed, context));
            }
        }

        Iterable<ConfigurationItem> existingDeployeds = filter(discovered, new Predicate<ConfigurationItem>() {
            @Override
            public boolean apply(ConfigurationItem input) {
                return input instanceof Deployed && deployeds.contains(input);
            }
        });

        return newArrayList((Iterable) existingDeployeds);
    }

    @SuppressWarnings("unchecked")
    public List<Deployed> cloneForInspection(List<Deployed> deployeds) {
        List<Deployed> clonedDeployeds = newArrayList();
        for (Deployed<?, ?> deployed : deployeds) {
            Descriptor descriptor = DescriptorRegistry.getDescriptor(deployed.getType());
            Deployed di = descriptor.newInstance();
            Iterable<PropertyDescriptor> inspectProperties = filter(descriptor.getPropertyDescriptors(), new Predicate<PropertyDescriptor>() {
                @Override
                public boolean apply(PropertyDescriptor input) {
                    return input.isInspectionProperty() || requiredForInspection.contains(input);
                }
            });
            di.setId(deployed.getId());
            di.setDeployable(deployed.getDeployable());
            di.setContainer(deployed.getContainer());
            for (PropertyDescriptor inspectProperty : inspectProperties) {
                inspectProperty.set(di, inspectProperty.get(deployed));
            }
            clonedDeployeds.add(di);
        }
        return clonedDeployeds;
    }

    protected boolean shouldCheckDeployedsDoNotExistBeforeCreating() {
        return deployMode == DeployMode.CREATE;
    }

    protected boolean shouldCreateDeployed() {
        return true;
    }

    protected boolean shouldCheckDeployedsCreatedCorrectly() {
        return true;
    }

    protected boolean shouldDestroyDeployeds() {
        return deployMode == DeployMode.CREATE;
    }

    protected boolean shouldCheckDeployedsDoNotExistAfterDelete() {
        return deployMode == DeployMode.CREATE;
    }

    protected abstract ConfigurationItem getCiToPerformInspectionOn(List<Type> typesToDiscover);
}
