package com.xebialabs.deployit.test.support;


import java.io.File;
import java.io.IOException;
import java.net.URL;
import java.util.List;
import java.util.Map;
import java.util.Set;

import org.jdom2.Document;
import org.jdom2.Element;
import org.jdom2.JDOMException;
import org.jdom2.input.SAXBuilder;
import org.jdom2.input.sax.XMLReaders;
import org.junit.rules.TemporaryFolder;
import com.google.common.base.Preconditions;
import com.google.common.base.Predicate;
import com.google.common.base.Splitter;
import com.google.common.collect.Iterables;

import com.xebialabs.deployit.booter.local.utils.Strings;
import com.xebialabs.deployit.plugin.api.reflect.Descriptor;
import com.xebialabs.deployit.plugin.api.reflect.DescriptorRegistry;
import com.xebialabs.deployit.plugin.api.reflect.PropertyDescriptor;
import com.xebialabs.deployit.plugin.api.reflect.Type;
import com.xebialabs.deployit.plugin.api.udm.ConfigurationItem;
import com.xebialabs.deployit.plugin.api.udm.Container;
import com.xebialabs.deployit.plugin.api.udm.Deployed;
import com.xebialabs.deployit.plugin.api.udm.artifact.Artifact;
import com.xebialabs.overthere.local.LocalFile;

import static com.google.common.base.Joiner.on;
import static com.google.common.base.Preconditions.checkArgument;
import static com.google.common.collect.Iterables.filter;
import static com.google.common.collect.Lists.newArrayList;
import static com.google.common.collect.Maps.newHashMap;
import static com.google.common.collect.Maps.newLinkedHashMap;
import static com.google.common.collect.Sets.newHashSet;
import static com.google.common.io.Files.copy;
import static com.google.common.io.Resources.newInputStreamSupplier;
import static com.xebialabs.overthere.util.OverthereUtils.getName;
import static org.hamcrest.CoreMatchers.equalTo;
import static org.junit.Assert.assertThat;

public class DeployedItestChangeSet {

    public enum DeployMode {
        CREATE,
        UPDATE_ONLY
    }

    private DeployMode deployMode = DeployMode.CREATE;

    private Map<String,Deployed> create = newLinkedHashMap();
    private Map<String,Deployed> modify = newLinkedHashMap();
    private List<Type> additionalTypesToDiscover = newArrayList();
    private Set<PropertyDescriptor> requiredForInspection = newHashSet();

    private File changesetXmlFile;
    private Container container;
    private ItestTopology topology;
    private TemporaryFolder folder;

    protected DeployedItestChangeSet(File changesetXmlFile,  Container container, ItestTopology topology, TemporaryFolder folder) {
        this.changesetXmlFile = changesetXmlFile;
        this.container = container;
        this.topology = topology;
        this.folder = folder;
    }

    public static DeployedItestChangeSet loadChangeSet(File changesetXmlFile, Container container, ItestTopology topology, TemporaryFolder folder) throws JDOMException, IOException {
        DeployedItestChangeSet cs = new DeployedItestChangeSet(changesetXmlFile, container, topology,folder);
        cs.init();
        return cs;
    }


    protected void init() throws JDOMException, IOException {
        SAXBuilder sb = new SAXBuilder(XMLReaders.NONVALIDATING);
        Document deployedProperties = sb.build(changesetXmlFile);

        Element rootElement = deployedProperties.getRootElement();

        setDeployMode(rootElement.getAttributeValue("deployMode"));

        String disableInTopology = rootElement.getAttributeValue("disableInTopology");
        if (Strings.isNotBlank(disableInTopology)) {
            Iterable<String> searchTags = Splitter.on(',').trimResults().split(disableInTopology);
            if (!topology.getTags().isEmpty()) {
                for (String searchTag : searchTags) {
                    if (topology.getTags().contains(searchTag)) {
                        //disable the test for this topology
                        return;
                    }
                }
            }
        }

        List<Element> deployedGroupElements = rootElement.getChildren("deployeds");
        for (Element deployedGroupElement : deployedGroupElements) {
            String additionalTypes = deployedGroupElement.getAttributeValue("additionalTypesToDiscover");
            if (Strings.isNotBlank(additionalTypes)) {
                for (String t : additionalTypes.split(",")) {
                    addAdditionalTypeToDiscover(t);
                }
            }


            String target = deployedGroupElement.getAttributeValue("target");
            Container targetContainer = container;

            if (target != null) {
                Type containerType = Type.valueOf(target);
                if (container.getType().instanceOf(containerType)) {
                    //proceed to evaluate deployed specified in this group.
                } else {
                    boolean alwaysDeployToTarget = "true".equals(deployedGroupElement.getAttributeValue("alwaysDeployToTarget"));
                    if (alwaysDeployToTarget) {
                        //This deployed must always be deployed to the specified target as it is a dependency.
                        targetContainer = topology.findFirstMatchingTarget(containerType);
                    } else {
                        //This deployment group is not applicable for the current container. Ignore.
                        continue;
                    }
                }
            }

            List<Element> deployedElements = deployedGroupElement.getChildren();
            File deployedFolder = folder.newFolder();
            for (Element each : deployedElements) {
                convertDeployedElementToDeployed(each, targetContainer, deployedFolder);
            }
        }

        associateDeployedIdsWithContainer();
    }

    private void convertDeployedElementToDeployed(Element deployedElement, Container targetContainer, File deployedFolder) throws IOException {
        Descriptor d = DescriptorRegistry.getDescriptor(deployedElement.getName());
        checkArgument(d.isAssignableTo(Deployed.class), "Type [%s] is not assignable to udm.Deployed", d.getType());

        Deployed item = d.newInstance();

        if (item instanceof Artifact) {
            Element fileNameElm = deployedElement.getChild("fileName");
            checkArgument(fileNameElm != null, "Type [%s] is an artifact and must specify fileName that can be resolved from the classpath", d.getType());
            File artifact = new File(deployedFolder, getName(fileNameElm.getValue()));
            assertThat("Artifact " + artifact + " should not yet exist", artifact.exists(), equalTo(false));
            URL aURL = getClass().getClassLoader().getResource(fileNameElm.getValue());
            copy(newInputStreamSupplier(aURL), artifact);
            ((Artifact) item).setFile(LocalFile.valueOf(artifact));
        }

        item.setId(Preconditions.checkNotNull(deployedElement.getAttributeValue("id"), "Element [%s] has no id attribute", deployedElement.getName()));
        item.setContainer(targetContainer);

        boolean isModification = "true".equals(deployedElement.getAttributeValue("modification"));

        if (isModification) {
            Preconditions.checkArgument(deployedToBeCreatedExists(item.getId()), "Element [%s] has modification='true' but the original deployed with id [%s] could not be found.", deployedElement.getName(),item.getId());
            item = cloneDeployed(getDeployedToCreate(item.getId()));
        }


        for (Element p : deployedElement.getChildren()) {
            if (p.getName().equals("fileName")) {
                continue;
            }
            PropertyDescriptor pd = Preconditions.checkNotNull(d.getPropertyDescriptor(p.getName()), "Property [%s.%s] does not exist", d.getType(), p.getName());
            if (p.getAttributeValue("inspectionProperty") != null && p.getAttributeValue("inspectionProperty").equals("true")) {
                requiredForInspection.add(pd);
            }
            switch (pd.getKind()) {
                case CI:
                    pd.set(item, resolveCiReference(p, pd));
                    break;
                case MAP_STRING_STRING:
                    Map<String, String> map = newHashMap();
                    for (Element e : p.getChildren("entry")) {
                        map.put(Preconditions.checkNotNull(e.getAttributeValue("key")), Preconditions.checkNotNull(e.getValue()));
                    }
                    pd.set(item, map);
                    break;
                case SET_OF_STRING:
                    Set<String> setOfString = newHashSet();
                    for (Element v : p.getChildren("value")) {
                        setOfString.add(v.getValue());
                    }
                    pd.set(item, setOfString);
                    break;
                case LIST_OF_STRING:
                    List<String> listOfString = newArrayList();
                    for (Element v : p.getChildren("value")) {
                        listOfString.add(v.getValue());
                    }
                    pd.set(item, listOfString);
                    break;
                case LIST_OF_CI:
                    List<ConfigurationItem> listOfCi = newArrayList();
                    for (Element v : p.getChildren("value")) {
                        listOfCi.add(resolveCiReference(v, pd));
                    }
                    pd.set(item, listOfCi);
                    break;
                case SET_OF_CI:
                    Set<ConfigurationItem> setOfCi = newHashSet();
                    for (Element v : p.getChildren("value")) {
                        setOfCi.add(resolveCiReference(v, pd));
                    }
                    pd.set(item, setOfCi);
                    break;
                default:
                    pd.set(item, p.getValue());
            }
        }

        if (isModification) {
            addDeployedToModify(item);
        } else {
            addDeployedToCreate(item);
        }
    }

    protected ConfigurationItem resolveCiReference(Element p, PropertyDescriptor pd) {
        String ciRefId = p.getValue().trim();
        if (deployedToBeCreatedExists(ciRefId)) {
            return getDeployedToCreate(ciRefId);
        }

        ciRefId = topology.replacePlaceholders(ciRefId);
        ConfigurationItem ciRef = topology.getItems().get(ciRefId);
        if (ciRef == null) {
            ciRef = DescriptorRegistry.getDescriptor(pd.getReferencedType()).newInstance();
            ciRef.setId(ciRefId);
        }

        return ciRef;
    }

    protected void setDeployMode(String deployModeAttribute) {
        if (deployModeAttribute == null) {
            return;
        }
        try {
            deployMode = DeployMode.valueOf(deployModeAttribute.toUpperCase());
        } catch(IllegalArgumentException e) {
            throw new IllegalArgumentException(String.format("Unsupported deployMode value '%s' on itest tag expected [%s]",
                    deployModeAttribute, on("|").join(newArrayList(DeployMode.values()))));
        }
    }

    protected void associateDeployedIdsWithContainer() {
        Iterable<Deployed> deployeds = Iterables.concat(getDeployedsToCreate(), getDeployedsToModify());
        for (Deployed deployed : deployeds) {
            Preconditions.checkNotNull(deployed.getContainer());
            deployed.setId(deployed.getContainer().getId() + "/" + deployed.getId());
        }
    }

    protected List<Deployed> cloneForInspection(List<Deployed> deployeds) {
        List<Deployed> clonedDeployeds = newArrayList();
        for (Deployed<?, ?> deployed : deployeds) {
            Descriptor descriptor = DescriptorRegistry.getDescriptor(deployed.getType());
            Deployed di = descriptor.newInstance();
            Iterable<PropertyDescriptor> inspectProperties = filter(descriptor.getPropertyDescriptors(), new Predicate<PropertyDescriptor>() {
                @Override
                public boolean apply(PropertyDescriptor input) {
                    return input.isInspectionProperty() || requiredForInspection.contains(input);
                }
            });
            di.setId(deployed.getId());
            di.setDeployable(deployed.getDeployable());
            di.setContainer(deployed.getContainer());
            for (PropertyDescriptor inspectProperty : inspectProperties) {
                inspectProperty.set(di, inspectProperty.get(deployed));
            }
            clonedDeployeds.add(di);
        }
        return clonedDeployeds;

    }

    protected Deployed cloneDeployed(Deployed deployed) {
        Descriptor descriptor = DescriptorRegistry.getDescriptor(deployed.getType());
        Deployed clonedDeployed = descriptor.newInstance();
        clonedDeployed.setId(deployed.getId());
        clonedDeployed.setDeployable(deployed.getDeployable());
        clonedDeployed.setContainer(deployed.getContainer());
        for (PropertyDescriptor inspectProperty : descriptor.getPropertyDescriptors()) {
            inspectProperty.set(clonedDeployed, inspectProperty.get(deployed));
        }
        return clonedDeployed;

    }

    public DeployMode getDeployMode() {
        return deployMode;
    }

    public List<Type> getAdditionalTypesToDiscover() {
        return additionalTypesToDiscover;
    }

    public boolean hasDeployedsToCreate() {
        return !create.isEmpty();
    }

    public boolean hasDeployedsToModify() {
        return !modify.isEmpty();
    }

    public void addAdditionalTypeToDiscover(String type) {
        additionalTypesToDiscover.add(Type.valueOf(type.trim()));
    }

    public void addDeployedToCreate(Deployed d) {
        create.put(d.getId(), d);
    }

    public void addDeployedToModify(Deployed d) {
        modify.put(d.getId(),d);
    }


    public Deployed getDeployedToCreate(String id) {
        return create.get(id);
    }

    public Deployed getDeployedToModify(String id) {
        return modify.get(id);
    }

    public boolean deployedToBeCreatedExists(String id) {
        return create.containsKey(id);
    }

    public List<Deployed> getDeployedsToCreate() {
        return newArrayList(create.values());
    }

    public List<Deployed> getDeployedsToModify() {
        return newArrayList(modify.values());
    }
}