package com.xebialabs.deployit.test.support;

import java.io.File;
import java.io.IOException;
import java.net.URISyntaxException;
import java.net.URL;
import java.nio.charset.Charset;
import java.util.ArrayList;
import java.util.Arrays;
import java.util.Iterator;
import java.util.List;
import java.util.Map;
import java.util.Set;

import org.hamcrest.CoreMatchers;
import org.jdom2.JDOMException;
import org.junit.After;
import org.junit.AfterClass;
import org.junit.Before;
import org.junit.BeforeClass;
import org.junit.Rule;
import org.junit.Test;
import org.junit.rules.TemporaryFolder;
import org.junit.runner.RunWith;
import org.junit.runners.Parameterized;
import org.junit.runners.Parameterized.Parameters;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;

import com.google.common.base.Function;
import com.google.common.base.Predicate;
import com.google.common.collect.Iterables;
import com.google.common.collect.Lists;
import com.google.common.io.Files;

import com.xebialabs.deployit.booter.local.LocalBooter;
import com.xebialabs.deployit.deployment.planner.DeltaSpecificationBuilder;
import com.xebialabs.deployit.plugin.api.deployment.specification.DeltaSpecification;
import com.xebialabs.deployit.plugin.api.flow.StepExitCode;
import com.xebialabs.deployit.plugin.api.reflect.Descriptor;
import com.xebialabs.deployit.plugin.api.reflect.DescriptorRegistry;
import com.xebialabs.deployit.plugin.api.reflect.PropertyDescriptor;
import com.xebialabs.deployit.plugin.api.reflect.Type;
import com.xebialabs.deployit.plugin.api.udm.ConfigurationItem;
import com.xebialabs.deployit.plugin.api.udm.Container;
import com.xebialabs.deployit.plugin.api.udm.Deployed;
import com.xebialabs.deployit.plugin.api.validation.ValidationMessage;
import com.xebialabs.deployit.test.deployment.DeployitTester;
import com.xebialabs.deployit.test.support.DeployedItestChangeSet.DeployMode;
import com.xebialabs.deployit.test.support.DeployedItestChangeSet.TestAction;
import com.xebialabs.overcast.host.CloudHost;
import com.xebialabs.overcast.host.CloudHostFactory;

import static com.google.common.base.Joiner.on;
import static com.google.common.base.Preconditions.checkArgument;
import static com.google.common.base.Preconditions.checkNotNull;
import static com.google.common.collect.Iterables.filter;
import static com.google.common.collect.Lists.newArrayList;
import static com.google.common.collect.Sets.newHashSet;
import static com.xebialabs.deployit.test.support.DeployedItestChangeSet.TestAction.CREATE;
import static com.xebialabs.deployit.test.support.DeployedItestChangeSet.TestAction.DESTROY;
import static com.xebialabs.deployit.test.support.DeployedItestChangeSet.TestAction.MODIFY;
import static com.xebialabs.deployit.test.support.DeployedItestChangeSet.TestAction.VERIFYABSENCE;
import static com.xebialabs.deployit.test.support.DeployedItestChangeSet.TestAction.VERIFYCREATE;
import static com.xebialabs.deployit.test.support.DeployedItestChangeSet.TestAction.VERIFYDESTROY;
import static com.xebialabs.deployit.test.support.DeployedItestChangeSet.TestAction.VERIFYMODIFY;
import static com.xebialabs.deployit.test.support.TestUtils.createDeployedApplication;
import static com.xebialabs.platform.test.TestUtils.createDeploymentPackage;
import static com.xebialabs.platform.test.TestUtils.createEnvironment;
import static java.lang.String.format;
import static org.hamcrest.CoreMatchers.equalTo;
import static org.hamcrest.MatcherAssert.assertThat;
import static org.junit.Assert.fail;
import static org.junit.Assume.assumeTrue;

/** @see <a href="https://intranet.xebia.com/confluence/display/Labs/Deployed+Itesting+with+xml">https://intranet.xebia.com/confluence/display/Labs/Deployed Itesting with xml</a>
 */
@SuppressWarnings("rawtypes")
@RunWith(Parameterized.class)
public abstract class DeployedItestBase {

    static {
        LocalBooter.bootWithoutGlobalContext();
    }

    @Rule
    public TemporaryFolder folder = new TemporaryFolder();

    private static final String DEPLOYED_ITEST_DIR = "deployed-test-scripts";

    @SuppressWarnings("unused")
    private String description;

    private File deployedChangeSetXmlFile;

    protected ItestTopology topology;

    protected Container container;
    private static DeployitTester tester;

    protected DeployedItestChangeSet deployedChangeSet;

    private static TestExecutionContext context;

    public DeployedItestBase(String description, File deployedChangeSetXmlFile, ItestTopology topology, Container container) {
        this.description = description;
        this.deployedChangeSetXmlFile = deployedChangeSetXmlFile;
        this.topology = topology;
        this.container = container;
    }

    @Parameters(name = "{0}")
    public static List<Object[]> getTargets() throws Exception {
        List<Object[]> constructorArgsList = new ArrayList<Object[]>();
        Map<String, ItestTopology> topologies = ItestTopology.load();

        for (File deployedItestFile : getDeployedItestFiles()) {
            String desc = "Test " + deployedItestFile.getName() + " on ";
            for (ItestTopology topology : topologies.values()) {
                if (ItestTopology.isItestEnabled(topology.getId(), topology.isEnabledByDefault())) {
                    for (Container container : topology.getTargets()) {
                        constructorArgsList.add(new Object[]{desc + container, deployedItestFile, topology, container});
                    }
                }
            }
        }
        return constructorArgsList;
    }

    @Before
    public void takeCareOfVagrantImages() {
        CloudHost cloudHost = CloudHostFactory.getCloudHost(topology.getId());
        if (cloudHost != null) {
            CloudHostRegistry registry = CloudHostRegistry.getInstance();
            registry.makeSureAllDownInTheEnd();
            registry.makeSureIsUp(cloudHost);
            topology.registerIp(cloudHost.getHostName());
        }
    }

    @Before
    public void prepareDeployedChangeSet() throws JDOMException, IOException {
        deployedChangeSet = DeployedItestChangeSet.loadChangeSet(deployedChangeSetXmlFile, container, topology, folder);
    }
    @After
    public void clearInspectionContext() {
        context.clearInspectionContext();
    }

    @BeforeClass
    public static void setup() {
        tester = new DeployitTester();
        context = new TestExecutionContext(DeployedItestBase.class, tester.repository());
    }

    @AfterClass
    public static void tearDown() {
        context.destroy();
        tester.shutdown();
    }

    private static Iterable<File> getDeployedItestFiles() throws URISyntaxException, IOException {
        File deployedItestDir = getDeployedItestDir();
        final List<String> enabledItests = getEnabledItests();
        return Iterables.filter(Arrays.asList(deployedItestDir.listFiles()), new Predicate<File>() {
            @Override
            public boolean apply(File input) {
                String name = input.getName();
                return (name.endsWith(".xml")) &&
                        (enabledItests.isEmpty() || enabledItests.contains(name));
            }
        });
    }

    private static File getDeployedItestDir() throws URISyntaxException {
        URL deployedItestDirUrl = Thread.currentThread().getContextClassLoader().getResource(DEPLOYED_ITEST_DIR);
        checkNotNull(deployedItestDirUrl);
        return new File(deployedItestDirUrl.toURI());
    }

    private static List<String> getEnabledItests() throws URISyntaxException, IOException {
        File deployedItestDir = getDeployedItestDir();
        File enabledItestsFile = new File(deployedItestDir, "enabled-itests");
        List<String> enabledItests = newArrayList();
        if (enabledItestsFile.exists()) {
            List<String> tests = newArrayList(filter(Files.readLines(enabledItestsFile, Charset.defaultCharset()), new Predicate<String>() {
                @Override
                public boolean apply(String input) {
                    return !input.startsWith("#");
                }
            }));
            enabledItests.addAll(tests);
        }
        return enabledItests;
    }

    @Test
    public void shouldCreateAndDestroyDeployed() throws Exception {
        assumeTrue(deployedChangeSet.hasDeployedsToCreate());

        if(deployedChangeSet.getTests().isEmpty()) {
            runSpecifiedTests(defaultTestsToRun());
        } else {
            runSpecifiedTests(deployedChangeSet.getTests());
        }
    }

    private void runSpecifiedTests(List<TestAction> tests) {
        System.out.println("\nTest plan to execute :");
        int count = 1;
        for (TestAction t : tests) {
            System.out.println(count++ + ") " + t.toString().toLowerCase());
        }

        count = 1;
        System.out.println("\nExecuting test plan...");
        for (TestAction t : tests) {
            System.out.println("\n" + count++ + ") " + t + "\n");
            switch (t) {
                case VERIFYABSENCE:
                case VERIFYDESTROY:
                    assertDeployedsDoNotExist(deployedChangeSet.getDeployedsToCreate());
                    break;
                case CREATE:
                    createDeployeds(deployedChangeSet.getDeployedsToCreate());
                    break;
                case VERIFYCREATE:
                    assertDeployedsWereCreatedCorrectly(deployedChangeSet.getDeployedsToVerifyCreate());
                    break;
                case MODIFY:
                    if(deployedChangeSet.hasDeployedsToModify()) {
                        modifyDeployeds(deployedChangeSet);
                    } else {
                        System.out.println("No modifications defined.");
                    }
                    break;
                case VERIFYMODIFY:
                    if(deployedChangeSet.hasDeployedsToModify()) {
                        assertDeployedsWereCreatedCorrectly(deployedChangeSet.getDeployedsToVerifyModify());
                    } else {
                        System.out.println("No modifications to verify.");
                    }
                    break;
                case DESTROY:
                    destroyDeployeds(deployedChangeSet.getDeployedsToCreate());
                    break;
                case NOOP:
                    break;
            }
            context.clearInspectionContext();
        }
        System.out.println("Executing test plan complete.");

    }

    private List<TestAction> defaultTestsToRun() {
        List<TestAction> tests = newArrayList();

        if (shouldCheckDeployedsDoNotExistBeforeCreating()) {
            tests.add(VERIFYABSENCE);
        }

        if (shouldCreateDeployed()) {
            tests.add(CREATE);
        }

        if (shouldCheckDeployedsCreatedCorrectly()) {
            tests.add(VERIFYCREATE);
        }

        if (shouldModifyDeployeds()) {
            tests.add(MODIFY);
            if (shouldCheckDeployedsModifiedCorrectly()) {
                tests.add(VERIFYMODIFY);
            }
        }

        if (shouldDestroyDeployeds()) {
            tests.add(DESTROY);
        }

        if (shouldCheckDeployedsDoNotExistAfterDelete()) {
            tests.add(VERIFYDESTROY);
        }

        return tests;
    }

    protected void createDeployeds(List<Deployed> deployeds) {
        DeltaSpecificationBuilder builder = new DeltaSpecificationBuilder().initial(createDeployedApplication(createDeploymentPackage("1.0"), createEnvironment(container)));
        for (Deployed d : deployeds) {
            List<ValidationMessage> validationErrors = DescriptorRegistry.getDescriptor(d.getType()).validate(d);
            if (validationErrors.isEmpty()) {
                builder.create(d);
            } else {
                fail("Deployed has validation errors: " + validationErrors);
            }
        }
        DeltaSpecification spec = builder.build();
        List<com.xebialabs.deployit.plugin.api.flow.Step> resolvedPlan = tester.resolvePlan(spec);
        StepExitCode result = DeployitTester.executePlan(resolvedPlan, context);
        assertThat("Create deployed step failed", result, CoreMatchers.is(StepExitCode.SUCCESS));
    }

    protected void modifyDeployeds(final DeployedItestChangeSet deployedChangeSet) {

        DeltaSpecificationBuilder builder = new DeltaSpecificationBuilder().upgrade(
            createDeployedApplication(com.xebialabs.platform.test.TestUtils.createDeploymentPackage("1.0"), createEnvironment(container)),
            createDeployedApplication(com.xebialabs.platform.test.TestUtils.createDeploymentPackage("2.0"), createEnvironment(container))
        );

        for (Deployed d : deployedChangeSet.getDeployedsToModify()) {
            List<ValidationMessage> validationErrors = DescriptorRegistry.getDescriptor(d.getType()).validate(d);
            if (validationErrors.isEmpty()) {
                builder.modify(deployedChangeSet.getDeployedToCreate(d.getName()), d);
            } else {
                fail("Deployed has validation errors: " + validationErrors);
            }
        }
        DeltaSpecification spec = builder.build();
        List<com.xebialabs.deployit.plugin.api.flow.Step> resolvedPlan = tester.resolvePlan(spec);
        StepExitCode result = DeployitTester.executePlan(resolvedPlan, context);
        assertThat("Modify deployed step failed", result, CoreMatchers.is(StepExitCode.SUCCESS));
    }


    protected void destroyDeployeds(List<Deployed> deployeds) {
        DeltaSpecificationBuilder builder = new DeltaSpecificationBuilder().undeploy(createDeployedApplication(createDeploymentPackage("1.0"), createEnvironment(container)));
        for (Deployed d : deployeds) {
            builder.destroy(d);
        }
        DeltaSpecification spec = builder.build();

        List<com.xebialabs.deployit.plugin.api.flow.Step> resolvedPlan = tester.resolvePlan(spec);
        StepExitCode result = DeployitTester.executePlan(resolvedPlan, context);
        assertThat("Destroy deployed step failed", result, CoreMatchers.is(StepExitCode.SUCCESS));
    }

    protected void assertDeployedsDoNotExist(List<Deployed> deployeds) {
        try {
            List<Deployed> actualDeployeds = inspectDeployeds(deployeds);

            if (actualDeployeds.size() > 0) {
                fail(format("Unexpected deployeds [%s]", on(",").join(actualDeployeds)));
            }
        } catch (RuntimeException e) {
            if (e.getMessage().contains("Step failed:") || e.getMessage().contains("CliScriptException")) {
                //That's ok, we want it! Success!
            } else {
                throw e;
            }
        }
    }

    protected void assertDeployedsWereCreatedCorrectly(List<Deployed> expectedDeployeds) {
        List<Deployed> actualDeployeds = inspectDeployeds(expectedDeployeds);

        if (actualDeployeds.size() != expectedDeployeds.size()) {
            fail(format("Expected deployeds [%s] to exist, but only found [%s]", on(",").join(expectedDeployeds), on(",").join(actualDeployeds)));
        }

        for (Deployed expected : expectedDeployeds) {
            int i = actualDeployeds.indexOf(expected);
            checkArgument(i != -1, "Cannot find [%s] in list of existing deployeds", expected);
            Deployed actual = actualDeployeds.get(i);
            assertDeployedWasCreatedCorrectly(expected, actual);
        }
    }

    @SuppressWarnings("unchecked")
    protected void assertDeployedWasCreatedCorrectly(ConfigurationItem expectedDeployed, ConfigurationItem actualDeployed) {
        assertThat("Expected actualDeployed and expectedDeployed to be of the same type", actualDeployed.getType(), equalTo(expectedDeployed.getType()));
        Descriptor dd = DescriptorRegistry.getDescriptor(expectedDeployed.getType());
        for (PropertyDescriptor pd : dd.getPropertyDescriptors()) {
            if (pd.isHidden() || pd.isPassword() || "deployable".equals(pd.getName())) {
                continue;
            }

            switch (pd.getKind()) {
                case MAP_STRING_STRING:
                    Map<String, String> expectedMap = (Map<String, String>) pd.get(expectedDeployed);
                    Map<String, String> actualMap = (Map<String, String>) pd.get(actualDeployed);
                    assertThat(String.format("Expected Map<String,String> property '%s' to be the same on CI '%s'", pd.getName(), expectedDeployed.getName()), actualMap, equalTo(expectedMap));
                    break;
                case SET_OF_STRING:
                    assertThat(String.format("Expected Set<String> property '%s' to be the same on CI", pd.getName(), expectedDeployed.getName()), (Set<String>) pd.get(actualDeployed), equalTo((Set<String>) pd.get(expectedDeployed)));
                    break;
                case LIST_OF_CI: {
                    List<ConfigurationItem> expectedCis = newArrayList((List<ConfigurationItem>) pd.get(expectedDeployed));
                    List<ConfigurationItem> actualCis = newArrayList((List<ConfigurationItem>) pd.get(actualDeployed));
                        assertThat(String.format("Expected List<CI> property '%s' to be of the same size on CI '%s' expected: %s actual: %s", pd.getName(), expectedDeployed.getName(), pd.get(expectedDeployed), pd.get(actualDeployed)), actualCis.size(), equalTo(expectedCis.size()));
                        Iterator<ConfigurationItem> expectedIterator = expectedCis.iterator();
                        Iterator<ConfigurationItem> actualIterator = actualCis.iterator();
                        while (actualIterator.hasNext() && expectedIterator.hasNext()) {
                            ConfigurationItem a = actualIterator.next();
                            ConfigurationItem e = expectedIterator.next();
                            assertDeployedWasCreatedCorrectly(e, a);
                        }
                    break;
                }
                case SET_OF_CI: {
                    Set<ConfigurationItem> expectedCis = newHashSet((Set<ConfigurationItem>) pd.get(expectedDeployed));
                    Set<ConfigurationItem> actualCis = newHashSet((Set<ConfigurationItem>) pd.get(actualDeployed));
                    assertThat(String.format("Expected Set<CI> property '%s' on CI '%s' to be of the same size expected: %s actual: %s", pd.getName(), expectedDeployed.getName(), pd.get(expectedDeployed), pd.get(actualDeployed)), actualCis.size(), equalTo(expectedCis.size()));
                    LOOP_OVER_EXPECTEDS: for(ConfigurationItem e: expectedCis) {
                        for(ConfigurationItem a: actualCis) {
                            // actual id ends in the expected id
                            if(a.getId().endsWith(e.getId())) {
                                assertDeployedWasCreatedCorrectly(e, a);
                                continue LOOP_OVER_EXPECTEDS;
                            }
                        }
                        fail(String.format("Expected Set<CI> property '%s' on CI '%s' to have element '%s'", pd.getName(), expectedDeployed.getName(), e.getId()));
                    }
                    break;
                }
                default:
                    Object inspectedValue = pd.get(actualDeployed);
                    Object expectedValue = pd.get(expectedDeployed);
                    assertThat(String.format("Property '%s' on CI '%s' expected '%s' but was '%s'", pd.getName(), expectedDeployed.getName(), expectedValue, inspectedValue), inspectedValue, equalTo(expectedValue));
            }
        }
    }

    @SuppressWarnings("unchecked")
    protected List<Deployed> inspectDeployeds(final List<Deployed> deployeds) {
        List<Type> typesToDiscover = newArrayList();
        typesToDiscover.addAll(deployedChangeSet.getAdditionalTypesToDiscover());
        typesToDiscover.addAll(Lists.transform(deployeds, new Function<Deployed, Type>() {
            public Type apply(Deployed input) {
                return input.getType();
            }
        }));

        List<ConfigurationItem> discovered = newArrayList();
        ConfigurationItem ciToPerformInspectionOn = getCiToPerformInspectionOn(typesToDiscover);

        if (ciToPerformInspectionOn != null) {
            //perform new style discovery
            discovered = tester.runInspectionTask(ciToPerformInspectionOn, context);
        } else {
            //perform old style discovery
            if (!deployedChangeSet.getAdditionalTypesToDiscover().isEmpty()) {
                throw new UnsupportedOperationException("The discovery of additional types is not supported using old style discovery.");
            }
            for (Deployed<?, ?> deployed : deployedChangeSet.cloneForInspection(deployeds)) {
                context.clearInspectionContext();
                List<ConfigurationItem> c = tester.runInspectionTask(deployed, context);
                discovered.addAll(c);
            }
        }

        Iterable<ConfigurationItem> existingDeployeds = filter(discovered, new Predicate<ConfigurationItem>() {
            @Override
            public boolean apply(ConfigurationItem input) {
                return input instanceof Deployed && deployeds.contains(input);
            }
        });

        return newArrayList((Iterable) existingDeployeds);
    }



    protected boolean shouldCheckDeployedsDoNotExistBeforeCreating() {
        return deployedChangeSet.getDeployMode() == DeployMode.CREATE;
    }

    protected boolean shouldCreateDeployed() {
        return true;
    }

    protected boolean shouldCheckDeployedsCreatedCorrectly() {
        return true;
    }

    protected boolean shouldCheckDeployedsModifiedCorrectly() {
        return true;
    }

    protected boolean shouldModifyDeployeds() {
        return true;
    }

    protected boolean shouldDestroyDeployeds() {
        return deployedChangeSet.getDeployMode()  == DeployMode.CREATE;
    }

    protected boolean shouldCheckDeployedsDoNotExistAfterDelete() {
        return deployedChangeSet.getDeployMode()  == DeployMode.CREATE;
    }

    protected ConfigurationItem getCiToPerformInspectionOn(List<Type> typesToDiscover) {
        return null;
    }

    protected final Logger logger = LoggerFactory.getLogger(this.getClass());
}
