package com.xebialabs.deployit.itest;


import java.util.Arrays;
import java.util.Collections;
import java.util.List;
import java.util.Map;

import com.google.common.base.Function;
import com.google.common.collect.Lists;

import com.xebialabs.deployit.deployment.planner.DeltaSpecificationBuilder;
import com.xebialabs.deployit.plugin.api.deployment.specification.DeltaSpecification;
import com.xebialabs.deployit.plugin.api.flow.Step;
import com.xebialabs.deployit.plugin.api.flow.StepExitCode;
import com.xebialabs.deployit.plugin.api.reflect.Type;
import com.xebialabs.deployit.plugin.api.udm.*;
import com.xebialabs.deployit.test.deployment.DeployitTester;
import com.xebialabs.deployit.test.support.TestExecutionContext;

import static com.google.common.collect.Lists.newArrayList;
import static com.xebialabs.deployit.test.deployment.DeltaSpecifications.createDeployedApplication;
import static com.xebialabs.platform.test.TestUtils.createEnvironment;
import static com.xebialabs.platform.test.TestUtils.id;
import static com.xebialabs.platform.test.TestUtils.newInstance;
import static org.hamcrest.CoreMatchers.is;
import static org.junit.Assert.assertThat;

public class ItestWizard {

    protected DeployitTester tester;

    public ItestWizard(final DeployitTester tester) {
        this.tester = tester;
    }

    public Deployed<?, ?> deployed(Deployable deployable, Container container, String type, Map<String, String> properties) {
        Deployed<?, ?> deployed = tester.generateDeployed(deployable, container, Type.valueOf(type));
        deployed.setId(deployable.getId());
        for (Map.Entry<String, String> entry : properties.entrySet()) {
            deployed.setProperty(entry.getKey(), entry.getValue());
        }
        return deployed;
    }

    public Deployed<?, ?> deployed(Deployable deployable, Container container, String type) {
        return deployed(deployable, container, type, Collections.<String, String> emptyMap());
    }


    public static DeployedApplication newDeployedApplication(String name, String version, Deployed<?, ?>... deployeds) {
        List<Deployable> deployables = Lists.transform(Arrays.asList(deployeds), new Function<Deployed<?, ?>, Deployable>() {
            @Override
            public Deployable apply(final Deployed<?, ?> d) {
                return d.getDeployable();
            }
        });

        DeploymentPackage pkg = newDeploymentPackage(name, version, deployables.toArray(new Deployable[]{}));

        Environment env = createEnvironment();

        DeployedApplication deployedApp = newInstance("udm.DeployedApplication");
        deployedApp.setVersion(pkg);
        deployedApp.setEnvironment(env);
        deployedApp.setId(id(env.getId(), name));

        deployedApp.addDeployeds(deployeds);


        return deployedApp;
    }
    public static DeployedApplication newDeployedApplication(String name, String version, Deployable... deployables) {
        DeploymentPackage pkg = newDeploymentPackage(name, version, deployables);

        Environment env = createEnvironment();

        DeployedApplication deployedApp = newInstance("udm.DeployedApplication");
        deployedApp.setVersion(pkg);
        deployedApp.setEnvironment(env);
        deployedApp.setId(id(env.getId(), name));

        return deployedApp;
    }

    public static DeploymentPackage newDeploymentPackage(String name, String version, Deployable... deployables) {
        Application app = newInstance("udm.Application");
        app.setId(id("Application", name));

        DeploymentPackage pkg = newInstance("udm.DeploymentPackage");
        pkg.setId(id(app.getId(), version));
        pkg.setApplication(app);
        for (Deployable d : deployables) {
            d.setId(id(app.getId(), d.getId()));
            pkg.addDeployable(d);
        }
        return pkg;
    }


    public void assertInitial(DeployedApplication app) {
        DeltaSpecificationBuilder builder = new DeltaSpecificationBuilder()
                .initial(createDeployedApplication(app.getVersion(), app.getEnvironment()));
        for (Deployed<?, ?> d : app.getDeployeds()) {
            builder.create(d);
        }
        assertPlan(builder.build());
    }


    public void assertUpgrade(Version newVersion, DeployedApplication app, Deployed<?, ?> prevDeployed, Deployed<?, ?> deployed) {
        DeltaSpecification spec = new DeltaSpecificationBuilder()
                .upgrade(app, createDeployedApplication(newVersion, app.getEnvironment()))
                .modify(prevDeployed, deployed).build();
        assertPlan(spec);
    }

    @SuppressWarnings("rawtypes")
    public void assertUpgrade(DeployedApplication oldVersion, DeployedApplication newVersion) {
        List<Deployed> deployeds1 = newArrayList(oldVersion.getDeployeds());
        Version newVersion1 = newVersion.getVersion();
        List<Deployed> deployeds2 = newArrayList(newVersion.getDeployeds());

        Deployable[] deployables = newVersion1.getDeployables().toArray(new Deployable[]{});

        DeployedApplication newDeployedApplication = newDeployedApplication(newVersion1.getApplication().getName(), newVersion1.getVersion(), deployables);

        DeltaSpecificationBuilder builder = new DeltaSpecificationBuilder().upgrade(newDeployedApplication, createDeployedApplication(newVersion1, newDeployedApplication.getEnvironment()));

        assertThat("This method can assert only upgrades when amound of deployeds is not changed", deployeds1.size(), is(deployeds2.size()));

        for (int i = 0; i < deployeds1.size(); i ++) {
            builder.modify(deployeds1.get(i), deployeds2.get(i));
        }

        assertPlan(builder.build());
    }

    public void assertUndeploy(DeployedApplication app) {
        DeltaSpecificationBuilder builder = new DeltaSpecificationBuilder().undeploy(app);
        for (Deployed<?, ?> deployed : app.getDeployeds()) {
            builder.destroy(deployed);
        }
        assertPlan(builder.build());
    }


    protected void assertPlan(final DeltaSpecification spec) {
        List<Step> steps = tester.resolvePlan(spec);

        TestExecutionContext ctx = new TestExecutionContext(ItestWizard.class);

        try {
            assertThat(DeployitTester.executePlan(steps, ctx), is(StepExitCode.SUCCESS));
        } finally {
            ctx.destroy();
        }
    }

}