/*
 * Copyright (c) 2010 XebiaLabs B.V. All rights reserved.
 *
 * Your use of XebiaLabs Software and Documentation is subject to the Personal
 * License Agreement.
 *
 * http://www.xebialabs.com/deployit-personal-edition-license-agreement
 *
 * You are granted a personal license (i) to use the Software for your own
 * personal purposes which may be used in a production environment and/or (ii)
 * to use the Documentation to develop your own plugins to the Software.
 * "Documentation" means the how to's and instructions (instruction videos)
 * provided with the Software and/or available on the XebiaLabs website or other
 * websites as well as the provided API documentation, tutorial and access to
 * the source code of the XebiaLabs plugins. You agree not to (i) lease, rent
 * or sublicense the Software or Documentation to any third party, or otherwise
 * use it except as permitted in this agreement; (ii) reverse engineer,
 * decompile, disassemble, or otherwise attempt to determine source code or
 * protocols from the Software, and/or to  (iii) copy the Software or
 * Documentation (which includes the source code of the XebiaLabs plugins). You
 * shall not create or attempt to create any derivative works from the Software
 * except and only to the extent permitted by law. You will preserve XebiaLabs'
 * copyright and legal notices on the Software and Documentation. XebiaLabs
 * retains all rights not expressly granted to You in the Personal License
 * Agreement.
 */

package com.xebialabs.deployit.plugin.jbossas.runbook;

import java.util.HashSet;
import java.util.List;
import java.util.Set;

import com.xebialabs.deployit.Change;
import com.xebialabs.deployit.ChangePlan;
import com.xebialabs.deployit.Step;
import com.xebialabs.deployit.ci.Deployment;
import com.xebialabs.deployit.mapper.artifact.ConfigurationFilesToHostMapper;
import com.xebialabs.deployit.mapper.artifact.LibrariesToHostMapper;
import com.xebialabs.deployit.mapper.artifact.SqlScriptToDatabaseMapper;
import com.xebialabs.deployit.plugin.jbossas.ci.JbossasServer;
import com.xebialabs.deployit.plugin.jbossas.mapper.EarToJbossasServerMapper;
import com.xebialabs.deployit.plugin.jbossas.mapper.JbossasDataSourceToJbossasServerMapper;
import com.xebialabs.deployit.plugin.jbossas.mapper.JbossasQueueToJbossasServerMapper;
import com.xebialabs.deployit.plugin.jbossas.mapper.JbossasTopicToJbossasServerMapper;
import com.xebialabs.deployit.plugin.jbossas.mapper.WarToJbossasServerMapper;
import com.xebialabs.deployit.plugin.jbossas.step.JbossasRestartServerStep;
import com.xebialabs.deployit.plugin.jbossas.step.JbossasWaitForDeploymentsToCompleteStep;
import com.xebialabs.deployit.util.SingleTypeHandlingRunBook;

/**
 * Supports the deployment, re-deployment and undeployment of an {@link com.xebialabs.deployit.ci.DeploymentPackage
 * Application Package} to a {@link com.xebialabs.deployit.plugin.jbossas.ci.JbossasServer JBoss Application Server}.
 * 
 * <h4>Conditions</h4>
 * 
 * Will trigger if the change plan contains the addition, modification or deletion of a {@link Deployment Deployment} CI
 * to an {@link com.xebialabs.deployit.ci.Environment Environment} CI which contains JBoss middleware CI's.
 * 
 * Note that Additions, Modifications and Deletions are always handled in one single flow.
 * 
 * <h4>Actions</h4>
 * 
 * <ol>
 * <li>Undeploy {@link com.xebialabs.deployit.ci.artifact.Ear EARs} from
 * {@link com.xebialabs.deployit.plugin.jbossas.ci.JbossasServer JBoss Application Server}
 * <li>Undeploy {@link com.xebialabs.deployit.ci.artifact.War WARs} from
 * {@link com.xebialabs.deployit.plugin.jbossas.ci.JbossasServer JBoss Application Server}
 * <li>Destroy {@link com.xebialabs.deployit.plugin.jbossas.ci.JbossasResource JBoss resources} from
 * {@link com.xebialabs.deployit.plugin.jbossas.ci.JbossasServer JBoss Application Server}
 * <li>Deploy {@link com.xebialabs.deployit.ci.artifact.Ear EARs} to
 * {@link com.xebialabs.deployit.plugin.jbossas.ci.JbossasServer JBoss Application Server}
 * <li>Deploy {@link com.xebialabs.deployit.ci.artifact.War WARs} to
 * {@link com.xebialabs.deployit.plugin.jbossas.ci.JbossasServer JBoss Application Server}
 * <li>Create {@link com.xebialabs.deployit.plugin.jbossas.ci.JbossasResource JBoss resources} from
 * {@link com.xebialabs.deployit.plugin.jbossas.ci.JbossasServer JBoss Application Server}
 * <li>Undeploy {@link com.xebialabs.deployit.ci.artifact.Libraries} from {@link com.xebialabs.deployit.ci.Host}
 * <li>Undeploy {@link com.xebialabs.deployit.ci.artifact.ConfigurationFiles} from
 * {@link com.xebialabs.deployit.ci.Host}
 * <li>Deploy {@link com.xebialabs.deployit.ci.artifact.Libraries} on {@link com.xebialabs.deployit.ci.Host}
 * <li>Deploy {@link com.xebialabs.deployit.ci.artifact.ConfigurationFiles} on {@link com.xebialabs.deployit.ci.Host}
 * </ol>
 * 
 */
public class JbossasDeploymentRunBook extends SingleTypeHandlingRunBook<Deployment> {

	public JbossasDeploymentRunBook() {
		super(Deployment.class);
	}

	@Override
	protected void resolve(Change<Deployment> change, ChangePlan changePlan, List<Step> steps) {
		WarToJbossasServerMapper warToServerMapper = new WarToJbossasServerMapper(change);
		EarToJbossasServerMapper earToServerMapper = new EarToJbossasServerMapper(change);
		JbossasDataSourceToJbossasServerMapper dataSourceToServerMapper = new JbossasDataSourceToJbossasServerMapper(change);
		JbossasQueueToJbossasServerMapper queueToServerMapper = new JbossasQueueToJbossasServerMapper(change);
		JbossasTopicToJbossasServerMapper topicToServerMapper = new JbossasTopicToJbossasServerMapper(change);
		LibrariesToHostMapper librariesMapper = new LibrariesToHostMapper(change);
		ConfigurationFilesToHostMapper configurationFilesToHostMapper = new ConfigurationFilesToHostMapper(change);

		Set<JbossasServer> affectedServers = new HashSet<JbossasServer>();
		affectedServers.addAll(warToServerMapper.getAffectedTargets());
		affectedServers.addAll(earToServerMapper.getAffectedTargets());
		affectedServers.addAll(dataSourceToServerMapper.getAffectedTargets());
		affectedServers.addAll(queueToServerMapper.getAffectedTargets());
		affectedServers.addAll(topicToServerMapper.getAffectedTargets());

		if (!isJbossasDeployment(warToServerMapper, earToServerMapper, dataSourceToServerMapper, queueToServerMapper, topicToServerMapper)) {
			return;
		}

		int lastStepCount = steps.size();
		earToServerMapper.generateDeletionSteps(steps);
		warToServerMapper.generateDeletionSteps(steps);
		topicToServerMapper.generateDeletionSteps(steps);
		queueToServerMapper.generateDeletionSteps(steps);
		dataSourceToServerMapper.generateDeletionSteps(steps);
		configurationFilesToHostMapper.generateDeletionSteps(steps);
		librariesMapper.generateDeletionSteps(steps);
		
		// no modification steps as they don't exist for JBoss
		
		SqlScriptToDatabaseMapper sqlMapper = new SqlScriptToDatabaseMapper(change);
		sqlMapper.generateAdditionSteps(steps);
		
		librariesMapper.generateAdditionSteps(steps);
		configurationFilesToHostMapper.generateAdditionSteps(steps);
		dataSourceToServerMapper.generateAdditionSteps(steps);
		queueToServerMapper.generateAdditionSteps(steps);
		topicToServerMapper.generateAdditionSteps(steps);
		warToServerMapper.generateAdditionSteps(steps);
		earToServerMapper.generateAdditionSteps(steps);

		if (steps.size() > lastStepCount) {
			restartServers(affectedServers, steps);
		}
	}

	private boolean isJbossasDeployment(WarToJbossasServerMapper warToServerMapper, EarToJbossasServerMapper earToServerMapper,
			JbossasDataSourceToJbossasServerMapper dataSourceToServerMapper, JbossasQueueToJbossasServerMapper queueToServerMapper,
			JbossasTopicToJbossasServerMapper topicToServerMapper) {
		Set<JbossasServer> allServers = new HashSet<JbossasServer>();
		allServers.addAll(warToServerMapper.getAllTargets());
		allServers.addAll(earToServerMapper.getAllTargets());
		allServers.addAll(dataSourceToServerMapper.getAllTargets());
		allServers.addAll(queueToServerMapper.getAllTargets());
		allServers.addAll(topicToServerMapper.getAllTargets());
		return !allServers.isEmpty();
	}

	private void restartServers(Set<JbossasServer> servers, List<Step> steps) {
		for (JbossasServer eachServer : servers) {
			if (eachServer.getRestartCommand() != null) {
				steps.add(new JbossasRestartServerStep(eachServer));
			}
		}
		steps.add(new JbossasWaitForDeploymentsToCompleteStep());
	}

}
