/*
 * Copyright (c) 2010 XebiaLabs B.V. All rights reserved.
 *
 * Your use of XebiaLabs Software and Documentation is subject to the Personal
 * License Agreement.
 *
 * http://www.xebialabs.com/deployit-personal-edition-license-agreement
 *
 * You are granted a personal license (i) to use the Software for your own
 * personal purposes which may be used in a production environment and/or (ii)
 * to use the Documentation to develop your own plugins to the Software.
 * "Documentation" means the how to's and instructions (instruction videos)
 * provided with the Software and/or available on the XebiaLabs website or other
 * websites as well as the provided API documentation, tutorial and access to
 * the source code of the XebiaLabs plugins. You agree not to (i) lease, rent
 * or sublicense the Software or Documentation to any third party, or otherwise
 * use it except as permitted in this agreement; (ii) reverse engineer,
 * decompile, disassemble, or otherwise attempt to determine source code or
 * protocols from the Software, and/or to  (iii) copy the Software or
 * Documentation (which includes the source code of the XebiaLabs plugins). You
 * shall not create or attempt to create any derivative works from the Software
 * except and only to the extent permitted by law. You will preserve XebiaLabs'
 * copyright and legal notices on the Software and Documentation. XebiaLabs
 * retains all rights not expressly granted to You in the Personal License
 * Agreement.
 */

package com.xebialabs.deployit.tck.manifest;

import static org.junit.Assert.assertTrue;

import java.io.Serializable;
import java.util.Collection;

import org.apache.commons.collections.CollectionUtils;
import org.apache.commons.collections.Predicate;

import com.xebia.ad.plugin.PluginLoader;
import com.xebia.ad.repository.ConfigurationItemTypeDescriptorRepository;
import com.xebialabs.deployit.ConfigurationItem;
import com.xebialabs.deployit.RunBook;

/**
 * Provides helper methods for unit tests to verify Deployit plugin manifests. Example:
 * <p>
 * 
 * <pre>
 * &#064;Test
 * public void testManifest() {
 * 	ManifestTester.testManifest(&quot;com.acme.plugin.package&quot;);
 * }
 * </pre>
 * 
 * This will attempt to load all Deployit manifests on the classpath and compare them to all items in the given package, verifying the following:
 * 
 * <ul>
 * <li>that the plugin loader can successfully process all the manifests on the classpath
 * <li>that all classes annotated with {@link ConfigurationItem @ConfigurationItem} are included in the loaded manifests
 * <li>that all implementations of {@link RunBook} are included in the loaded manifests
 * </ul>
 */
public class ManifestTester {

	public static void testManifest(String basePackage) {
		PluginLoader pluginLoader = new PluginLoader();
		pluginLoader.repository = new ConfigurationItemTypeDescriptorRepository();
		pluginLoader.loadPluginsWithoutDisabling();
		assertTrue("Expected plugin loader to not have any loading errors", pluginLoader.getLoadingErrors().isEmpty());

		testConfigurationItemsAreMentionedInManifest(basePackage, pluginLoader);
		testRunBooksAreMentionedInManifest(basePackage, pluginLoader);
	}

	private static void testConfigurationItemsAreMentionedInManifest(String basePackage, PluginLoader pluginLoader) {
		Collection<Class<?>> ciTypesLoadedByPluginLoader = pluginLoader.getConfigurationItemTypes();
		ConfigurationItemScanner scanner = new ConfigurationItemScanner();
		Collection<Class<? extends Serializable>> ciClasses = scanner.getComponentClasses(basePackage);
		for (Class<? extends Serializable> eachCiClass : ciClasses) {
			assertTrue("Expected configuration item " + eachCiClass.getName() + " to be mentioned in a MANIFEST.MF", ciTypesLoadedByPluginLoader
					.contains(eachCiClass));
			System.out.println("Found configuration item " + eachCiClass.getName() + " in a MANIFEST.MF");
		}
	}

	private static void testRunBooksAreMentionedInManifest(String basePackage, PluginLoader pluginLoader) {
		Collection<RunBook> runBooksLoadedByPluginLoader = pluginLoader.getRunBooks();
		RunBookScanner scanner = new RunBookScanner();
		Collection<Class<? extends RunBook>> runBookClasses = scanner.getComponentClasses(basePackage);
		for (final Class<? extends RunBook> eachRunBookClass : runBookClasses) {
			boolean x = CollectionUtils.exists(runBooksLoadedByPluginLoader, new Predicate() {
				public boolean evaluate(Object arg0) {
					return arg0.getClass() == eachRunBookClass;
				}
			});
			assertTrue("Expected run book " + eachRunBookClass.getName() + " to be mentioned in a MANIFEST.MF", x);
			System.out.println("Found run book " + eachRunBookClass.getName() + " in a MANIFEST.MF");
		}
	}

}
