package com.xebialabs.deployit.plugin.steps

import java.io.InputStream
import java.net.URL

import com.xebialabs.deployit.plugin.api.flow.ExecutionContext
import com.xebialabs.overthere.OverthereFile
import com.xebialabs.overthere.util.OverthereUtils
import com.xebialabs.platform.sugar.ResourcesSugar

trait CopyFileSupport extends ResourcesSugar { self: TargetHostSupport =>

  private[steps] def mkTargetDirsIfRequired(targetPath: String, createTargetPath: Boolean = true)(implicit ctx: ExecutionContext, targetHostEnvironment: HostEnvironment) = {
    val targetDirectory: OverthereFile = remoteFile(targetPath).getParentFile
    if (!targetDirectory.exists()) {
      if (createTargetPath) {
        ctx.logOutput(s"Creating path $targetDirectory on host $targetHost")
        targetDirectory.mkdirs()
      } else {
        ctx.logError(s"Path $targetDirectory on host $targetHost does not exist.")
        throw ExitStep
      }
    }
  }

  private[steps] def deleteTargetFileIfPresent(targetPath: String)(implicit ctx: ExecutionContext, targetHostEnvironment: HostEnvironment) = if (remoteFile(targetPath).exists()) {
    ctx.logOutput(s"Deleting $targetPath since it already exists on host $targetHost")
    remoteFile(targetPath).deleteRecursively()
  }

  private[steps] def copyFileToTarget(sourceFile: OverthereFile, targetPath: String)(implicit ctx: ExecutionContext, targetHostEnvironment: HostEnvironment) {
    ctx.logOutput(s"Copying ${sourceFile.getPath} to $targetPath on host $targetHost")
    sourceFile.copyTo(remoteFile(targetPath))
  }

  private[steps] def copyUrlToTarget(url: URL, targetPath: String)(implicit ctx: ExecutionContext, targetHostEnvironment: HostEnvironment) {
    ctx.logOutput(s"Copying resource $url to $targetPath on host $targetHost")
    val inputStream: InputStream = url.openStream()
    try {
      OverthereUtils.write(Stream.continually(inputStream.read).takeWhile(-1 !=).map(_.toByte).toArray, remoteFile(targetPath))
    } finally {
      inputStream.close()
    }
  }

  private[steps] case object ExitStep extends Exception
}
