package com.xebialabs.deployit.plugin.steps

import java.util
import java.util.{Map => JMap}

import com.xebialabs.deployit.plugin.api.flow._
import com.xebialabs.deployit.plugin.api.rules.{RulePostConstruct, StepMetadata, StepParameter, StepPostConstructContext}
import grizzled.slf4j.Logger
import com.xebialabs.platform.script.jython._
import com.xebialabs.platform.script.jython.ScriptSource.byResource

@StepMetadata(name = "jython")
class JythonStep extends BaseStep with PreviewStep {

  @transient private lazy val logger = Logger(getClass)

  @transient private lazy val jython = new JythonSupport{}

  @StepParameter(description = "Path to the Python script to execute (relative to XL Deploy's classpath)")
  private val scriptPath: String = ""

  @StepParameter(description = "Dictionary with keys as variable name and values as variable that are passed to the Python script", required = false, calculated = true)
  private var jythonContext: JMap[String, Any] = new util.HashMap[String, Any]()

  @RulePostConstruct
  def doPostConstruct(ctx: StepPostConstructContext): Unit = {
    calculateOrder(ctx)
    calculateDescription(ctx)
    jythonContext = ContextHelper.defaultContext(ctx, jythonContext)
  }

  override def execute(executionContext: ExecutionContext): StepExitCode = {
    try {
      executeScript(executionContext)
    } catch {
      case e: JythonException => handleError(executionContext, e)
    }
  }

  private def executeScript(executionContext: ExecutionContext): StepExitCode = {
    import scala.collection.convert.wrapAll._
    implicit val scriptContext = createJythonContext(executionContext, Bindings.xlDeployApiServices ++ (jythonContext + ("context" -> executionContext)).toMap )
    jython.executeScript(byResource(scriptPath))
    StepExitCode.SUCCESS
  }

  private def createJythonContext(executionContext: ExecutionContext, variables: Map[String, Any]) = {
    JythonContext.withLibrariesAndFactory(Syntactic.loggerLib +: Syntactic.wrapperCodeWithLib(variables.keys)) {
      val scriptContext = variables.toScriptContext
      scriptContext.setWriter(new ConsumerWriter((text) => executionContext.logOutput(text)))
      scriptContext
    }
  }

  private def handleError(ctx: ExecutionContext, e: JythonException): StepExitCode = {
    logger.error(e.getMessage, e)
    ctx.logError(e.getMessage, e)
    StepExitCode.FAIL
  }

  override def getPreview: Preview = {
    Preview.withSourcePathAndContents(scriptPath, byResource(scriptPath).scriptContent)
  }
}
