package com.xebialabs.deployit.plugin.steps

import java.util
import java.util.{List => JList}

import com.xebialabs.deployit.plugin.api.flow._
import com.xebialabs.deployit.plugin.api.rules.{RulePostConstruct, StepMetadata, StepParameter, StepPostConstructContext}
import com.xebialabs.deployit.plugin.api.udm.artifact.Artifact
import com.xebialabs.overthere.{CmdLine, OperatingSystemFamily, OverthereFile}

import scala.collection.convert.wrapAsScala._
import scala.io.Source.fromInputStream

@StepMetadata(name = "os-script")
class OsScriptStep(context: util.Map[String, Any] = new util.HashMap[String, Any]())
  extends BaseStep with TargetHostSupport with FreemarkerSupport with CopyFileSupport with PreviewStep with StageableStep {

  def this() = this(new util.HashMap[String, Any]())

  freemarkerContext = context

  @StepParameter(name = "script", description = "Script base name that will be executed on the target host")
  protected[steps] val scriptBaseName: String = null

  @StepParameter(description = "Resources that are attached to the script and uploaded to the target host", required = false)
  protected[steps] val classpathResources: JList[String] = new util.ArrayList[String]

  @StepParameter(name = "uploadArtifacts", description = "If true, every artifact presented in the freemarker context will be uploaded to the working directory of the script", required = false)
  protected[steps] val uploadArtifactsInFreemarkerContext: Boolean = true

  private var stagedArtifacts: Vector[StagedArtifact] = Vector()

  @RulePostConstruct
  def doPostConstruct(ctx: StepPostConstructContext): Unit = {
    calculateOrder(ctx)
    calculateDescription(ctx)
    calculateTargetHost(ctx)
    calculateFreemarkerContext(ctx)
  }

  override def requestStaging(ctx: StagingContext) = if (Option(uploadArtifactsInFreemarkerContext).getOrElse(true)) {
    stagedArtifacts = freemarkerContext.values().collect { case a: Artifact => a}.map(a => StagedArtifact(ctx.stageArtifact(a, targetHost), a)).toVector
  }

  override protected[steps] def executeOnTargetHost(implicit ctx: ExecutionContext, targetHostEnvironment: HostEnvironment): StepExitCode = {
    val targetScriptName = processScript
    processClasspathResources
    uploadArtifacts
    executeScript(targetScriptName)
  }

  private[steps] def processScript(implicit executionContext: ExecutionContext, targetHostEnvironment: HostEnvironment) = {
    val resolvedScriptResource = resolveScriptResource
    processAndUploadResource(resolvedScriptResource)
    resolvedScriptResource.stripSuffix(".ftl")
  }

  private[steps] def processClasspathResources(implicit executionContext: ExecutionContext, targetHostEnvironment: HostEnvironment) = {
    Option(classpathResources).getOrElse(new util.ArrayList[String]()) foreach processAndUploadResource
  }

  private[steps] def processAndUploadResource(resourceName: String)(implicit executionContext: ExecutionContext, targetHostEnvironment: HostEnvironment) = {
    val targetFile = fileInWorkDir(resourceName.stripSuffix(".ftl"))

    mkTargetDirsIfRequired(targetFile.getPath)

    if (resourceName.endsWith(".ftl")) {
      renderFreemarkerTo(resourceName, targetFile)
    } else {
      copyUrlToTarget(resource(resourceName), targetFile.getPath)
    }
  }

  private[steps] def uploadArtifacts(implicit executionContext: ExecutionContext, targetHostEnvironment: HostEnvironment) {
    stagedArtifacts.foreach { sa =>
      val targetPath = fileInWorkDir(sa.artifact.getFile.getName).getPath
      copyFileToTarget(sa.stagedFile.get(targetHostEnvironment.connection, executionContext), targetPath)
    }
  }

  private[steps] def executeScript(scriptResource: String)(implicit ctx: ExecutionContext, targetHostEnvironment: HostEnvironment): StepExitCode = {
    val executable: OverthereFile = fileInWorkDir(scriptResource)
    executable.setExecutable(true)
    executeCommand(CmdLine.build(executable.getPath))
  }

  private[steps] def resolveScriptResource = {
    val possibleScriptsWithUrls = (targetOs match {
      case OperatingSystemFamily.WINDOWS => List("", ".bat", ".cmd", ".bat.ftl", ".cmd.ftl")
      case _ => List("", ".sh", ".sh.ftl")
    }).map(scriptBaseName + _).map(r => r -> resources(r)).toMap
    val allFoundResources = possibleScriptsWithUrls.values.flatten
    require(allFoundResources.size <= 1, s"Ambiguous resource specification: asked for $scriptBaseName and these are found: [${allFoundResources.mkString(", ")}]")
    require(allFoundResources.size != 0, s"No candidates found for script basename $scriptBaseName")
    possibleScriptsWithUrls.filterNot(_._2.isEmpty).head._1
  }

  override lazy val getPreview: Preview = {
    val scriptPath = resolveScriptResource
    val scriptContent = if (scriptPath.endsWith(".ftl"))
      processWithFreemarker(scriptPath, maskPassword = true)
    else
      fromInputStream(resourceAsInputStream(scriptPath)).getLines().mkString("\n")
    Preview.withSourcePathAndContents(scriptPath, scriptContent)
  }

  private case class StagedArtifact(stagedFile: StagedFile, artifact: Artifact) extends Serializable
}
