package com.xebialabs.deployit.plugin.steps

import java.io.StringWriter
import java.util
import java.util.{HashMap => JHashMap, Map => JMap}

import com.xebialabs.deployit.plugin.api.flow.ExecutionContext
import com.xebialabs.deployit.plugin.api.rules.{StepParameter, StepPostConstructContext}
import com.xebialabs.deployit.plugin.freemarker.ConfigurationHolder.configuration
import com.xebialabs.deployit.plugin.freemarker.sanitizer.OSScriptSanitizer
import com.xebialabs.overthere.OverthereFile
import scala.collection.convert.wrapAll._

trait FreemarkerSupport {

  @StepParameter(description = "Dictionary that contains all values available in the template", required = false, calculated = true)
  private[steps] var freemarkerContext: JMap[String, Any] = new JHashMap[String, Any]()

  private[steps] def calculateFreemarkerContext(ctx: StepPostConstructContext) {
    freemarkerContext = ContextHelper.defaultContext(ctx, freemarkerContext)
  }

  private[steps] def processWithFreemarker(templateResource: String, maskPassword: Boolean = false, artifactFiles: Map[ArtifactKey, OverthereFile] = Map.empty): String = {
    val writer: StringWriter = new StringWriter()
    val contextWithSanitizer: Any = addScriptSanitizer(freemarkerContext, templateResource)
    configuration(maskPasswords = maskPassword, artifactFiles).getTemplate(templateResource).process(contextWithSanitizer, writer)
    writer.toString
  }

  private[steps] def addScriptSanitizer(context: JMap[String, Any], templateName: String) = {
    new util.HashMap[String, Any](context + (OSScriptSanitizer.SANITIZE_EXPRESSION_AUTO ->
      OSScriptSanitizer.createOSSpecificSanitizer(templateName)))
  }

  private[steps] def renderFreemarkerTo(templateResource: String, destination: OverthereFile, artifactFiles: Map[ArtifactKey, OverthereFile] = Map.empty)(implicit ctx: ExecutionContext): Unit = {
    val out = destination.getOutputStream
    try {
      ctx.logOutput(s"Processing freemarker template $templateResource to ${destination.getPath}")
      out.write(processWithFreemarker(templateResource, false, artifactFiles).getBytes)
    } finally {
      out.close()
    }
  }
}
