package com.xebialabs.deployit.plugin.cloud.ci;

import java.util.*;
import com.google.common.base.Predicate;
import com.google.common.collect.ImmutableList;

import com.xebialabs.deployit.plugin.api.flow.Step;
import com.xebialabs.deployit.plugin.api.reflect.Type;
import com.xebialabs.deployit.plugin.api.udm.ControlTask;
import com.xebialabs.deployit.plugin.api.udm.Metadata;
import com.xebialabs.deployit.plugin.api.udm.Property;
import com.xebialabs.deployit.plugin.api.udm.base.BaseConfigurationItem;
import com.xebialabs.deployit.plugin.cloud.step.*;
import com.xebialabs.deployit.plugin.cloud.step.plan.SingleStepFilter;
import com.xebialabs.deployit.plugin.cloud.step.plan.StepOrderComparator;
import com.xebialabs.deployit.plugin.cloud.step.rule.SingleEnvCheck;
import com.xebialabs.deployit.plugin.cloud.util.CiParser;
import com.xebialabs.deployit.plugin.cloud.util.InstanceDescriptorResolver;

import static com.google.common.collect.Collections2.filter;
import static com.google.common.collect.Iterables.concat;
import static com.google.common.collect.Lists.newArrayList;
import static com.google.common.collect.Maps.newHashMap;
import static java.util.Collections.sort;

@SuppressWarnings("serial")
@Metadata(description = "Cloud environment template", root = Metadata.ConfigurationItemRoot.CONFIGURATION)
public class EnvironmentTemplate extends BaseConfigurationItem {

    private static InstanceDescriptorResolver descriptorResolver = new InstanceDescriptorResolver();

    private static CiParser ciParser = new CiParser();

    // CI properties

    @Property(required = false, size = Property.Size.LARGE, label = "Description of the template")
    private String description;

    @Property(required = true, label = "Host templates")
    private List<BaseHostTemplate> hostTemplates;


    @Property(required = true, size = Property.Size.LARGE, label = "Environment descriptor", description = "Freemarker template of XML which describes environment")
    private String xmlDescriptor;

    // Getters & setters

    public String getDescription() {
        return description;
    }

    public void setDescription(final String description) {
        this.description = description;
    }

    public void setHostTemplates(final List<BaseHostTemplate> hostTemplates) {
        this.hostTemplates = hostTemplates;
    }

    public void setXmlDescriptor(final String xmlDescriptor) {
        this.xmlDescriptor = xmlDescriptor;
    }

    public List<BaseHostTemplate> getHostTemplates() {
        return hostTemplates;
    }


    public String getXmlDescriptor() {
        return xmlDescriptor;
    }

    @ControlTask(label = "Instantiate environment", parameterType = "cloud.CloudEnvironmentParameters", description = "Instantiate environment and all hosts, which templates are linked to this environment template")
    public List<? extends Step> instantiate(CloudEnvironmentParameters parameters) {

        ArrayList<Step> steps = newArrayList();

        steps.add(new CheckParametersStep(parameters));

        final Map<Type, Collection<? extends Step>> waitStepsAdded = newHashMap();

        if (!getHostTemplates().isEmpty()) {

            int seq = 1;

            for (final BaseHostTemplate template : getHostTemplates()) {

                final Collection<? extends Step> presentSteps = waitStepsAdded.get(template.getType());

                Collection<? extends Step> stepsToAdd = filter(template.produceCreateSteps(getName(), seq), new SingleStepFilter(presentSteps));

                steps.addAll(stepsToAdd);

                waitStepsAdded.put(template.getType(), presentSteps == null ? stepsToAdd : ImmutableList.copyOf(concat(presentSteps, stepsToAdd)));
                seq++;
            }

            steps.add(new RegisterInstancesStep(descriptorResolver, ciParser, parameters.getHostsPath()));
        }

        steps.add(new RegisterEnvironmentStep(descriptorResolver, ciParser, this, parameters.getEnvironmentId()));

        sort(steps, new StepOrderComparator());

        return steps;
    }

    @ControlTask(label = "Validate descriptor", description = "Validate XML descriptor of the environment template")
    public List<? extends Step> validateEnvironmentDescriptor() {

        HashMap<Object,Object> scope = newHashMap();
        scope.put("environmentId", "Environments/descriptorTestEnv");
        scope.put("hosts", Collections.emptyList());
        scope.put("environmentTemplate", this);

        return newArrayList(new ValidateDescriptorStep(descriptorResolver, ciParser, scope, getXmlDescriptor(), new SingleEnvCheck()));
    }

}
