package com.xebialabs.deployit.plugin.cmd.deployed;

import java.util.Set;

import com.xebialabs.deployit.plugin.api.deployment.planning.Create;
import com.xebialabs.deployit.plugin.api.deployment.planning.DeploymentPlanningContext;
import com.xebialabs.deployit.plugin.api.deployment.planning.Destroy;
import com.xebialabs.deployit.plugin.api.deployment.planning.Modify;
import com.xebialabs.deployit.plugin.api.deployment.specification.Delta;
import com.xebialabs.deployit.plugin.api.deployment.specification.Operation;
import com.xebialabs.deployit.plugin.api.reflect.PropertyDescriptor;
import com.xebialabs.deployit.plugin.api.reflect.Type;
import com.xebialabs.deployit.plugin.api.udm.Metadata;
import com.xebialabs.deployit.plugin.api.udm.Property;
import com.xebialabs.deployit.plugin.api.udm.base.BaseDeployed;
import com.xebialabs.deployit.plugin.cmd.ci.Command;
import com.xebialabs.deployit.plugin.cmd.step.ExecuteCommandStep;
import com.xebialabs.deployit.plugin.cmd.step.NoCommandStep;
import com.xebialabs.deployit.plugin.file.File;
import com.xebialabs.deployit.plugin.overthere.Host;

import static com.google.common.collect.Sets.newHashSet;

@SuppressWarnings("serial")
@Metadata(description = "Command deployed to a Host")
public class DeployedCommand extends BaseDeployed<Command, Host> {

    @Property(description = "Order of the command", defaultValue = "50")
    private int order = 50;

    @Property(required = false, description = "Artifacts that the command depends on")
    private Set<File> dependencies = newHashSet();

    @Property(required = true, size = Property.Size.LARGE, description = "Command line to execute on host. Dependent artifacts can be referred to using ${artifact name}.")
    private String commandLine;

    @Property(required = false, description = "Command to execute when undeploying command")
    private Command undoCommand;

    @Property(required = false, description = "Indicates whether the undoCommand should be run on an upgrade")
    private boolean runUndoCommandOnUpgrade = true;

    @Property(required = false, isTransient = true, description = "Forces the command to be rerun.")
    private boolean rerunCommand;

    @Create
    public void executeCreateCommand(DeploymentPlanningContext ctx, Delta delta) {
        ctx.addStepWithCheckpoint(new ExecuteCommandStep(order, this), delta);
    }

    @Modify
    public void executeModifyCommand(DeploymentPlanningContext ctx, Delta delta) {
        if (undoCommand != null && runUndoCommandOnUpgrade) {
            DeployedCommand deployedUndoCommand = createDeployedUndoCommand();
            ctx.addStepWithCheckpoint(new ExecuteCommandStep(undoCommand.getOrder(), deployedUndoCommand), delta, Operation.DESTROY);
        }

        ctx.addStepWithCheckpoint(new ExecuteCommandStep(order, this), delta);
    }

    private DeployedCommand createDeployedUndoCommand() {
        DeployedCommand deployedUndoCommand = Type.valueOf(DeployedCommand.class).getDescriptor().newInstance(undoCommand.getId());
        deployedUndoCommand.setDeployable(undoCommand);
        deployedUndoCommand.setContainer(getContainer());

        for (PropertyDescriptor property : undoCommand.getType().getDescriptor().getPropertyDescriptors()) {
            if (deployedUndoCommand.hasProperty(property.getName())) {
                deployedUndoCommand.setProperty(property.getName(), undoCommand.getProperty(property.getName()));
            }
        }

        return deployedUndoCommand;
    }


    @Destroy
    public void destroyCommand(DeploymentPlanningContext ctx, Delta delta) {
        if (undoCommand != null) {
            DeployedCommand deployedUndoCommand = createDeployedUndoCommand();
            ctx.addStepWithCheckpoint(new ExecuteCommandStep(undoCommand.getOrder(), deployedUndoCommand), delta);
        } else {
            ctx.addStepWithCheckpoint(new NoCommandStep(order, this), delta);
        }
    }

    public Set<File> getDependencies() {
        return dependencies;
    }

    public void setDependencies(Set<File> dependencies) {
        this.dependencies = dependencies;
    }

    public int getOrder() {
        return order;
    }

    public void setOrder(int order) {
        this.order = order;
    }

    public String getCommandLine() {
        return commandLine;
    }

    public void setCommandLine(String commandLine) {
        this.commandLine = commandLine;
    }

    public Command getUndoCommand() {
        return undoCommand;
    }

    public void setUndoCommand(Command undoCommand) {
        this.undoCommand = undoCommand;
    }

    public boolean isRunUndoCommandOnUpgrade() {
        return runUndoCommandOnUpgrade;
    }

    public void setRunUndoCommandOnUpgrade(boolean runUndoCommandOnUpgrade) {
        this.runUndoCommandOnUpgrade = runUndoCommandOnUpgrade;
    }

    public boolean isRerunCommand() {
        return rerunCommand;
    }

    public void setRerunCommand(boolean rerunCommand) {
        this.rerunCommand = rerunCommand;
    }
}
