/*
 * @(#)CheckRequiredChangeRequest.java     26 Aug 2011
 *
 * Copyright © 2010 Andrew Phillips.
 *
 * ====================================================================
 * Licensed under the Apache License, Version 2.0 (the "License");
 * you may not use this file except in compliance with the License.
 * You may obtain a copy of the License at
 *
 * http://www.apache.org/licenses/LICENSE-2.0
 *
 * Unless required by applicable law or agreed to in writing, software
 * distributed under the License is distributed on an "AS IS" BASIS,
 * WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or 
 * implied.
 * See the License for the specific language governing permissions and
 * limitations under the License.
 * ====================================================================
 */
package com.xebialabs.deployit.plugins.releaseauth.planning;

import static com.google.common.collect.Collections2.filter;
import static com.google.common.collect.Collections2.transform;
import static com.xebialabs.deployit.plugins.releaseauth.ConditionVerifier.validateReleaseConditions;
import static java.lang.Boolean.TRUE;

import java.util.Collection;
import java.util.HashSet;
import java.util.List;
import java.util.Set;

import com.google.common.base.Function;
import com.google.common.base.Predicate;
import com.google.common.collect.ImmutableList;
import com.google.common.collect.ImmutableList.Builder;
import com.xebialabs.deployit.plugin.api.deployment.execution.DeploymentStep;
import com.xebialabs.deployit.plugin.api.deployment.planning.DefaultOrders;
import com.xebialabs.deployit.plugin.api.deployment.planning.PrePlanProcessor;
import com.xebialabs.deployit.plugin.api.deployment.specification.DeltaSpecification;
import com.xebialabs.deployit.plugin.api.reflect.PropertyDescriptor;
import com.xebialabs.deployit.plugin.api.udm.DeployedApplication;
import com.xebialabs.deployit.plugin.api.udm.Environment;
import com.xebialabs.deployit.plugins.releaseauth.ConditionVerifier.VerificationResult;
import com.xebialabs.deployit.plugins.releaseauth.ConditionVerifier.ViolatedCondition;
import com.xebialabs.deployit.plugins.releaseauth.step.LogReleaseConditionsStep;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;

public class CheckReleaseConditionsAreMet {
	public static final String ENV_RELEASE_CONDITIONS_PROPERTY = "releaseConditions";
	private static final String ENV_RECHECK_CONDITIONS_PROPERTY = "recheckConditionsAtDeploymentTime";
	private static final String ENV_RECHECK_CONDITIONS_ORDER_PROPERTY = "recheckConditionsAtDeploymentTimeOrder";
	private static final String RELEASE_CHECKLIST_CATEGORY_NAME = "Deployment Checklist";

	private static final List<DeploymentStep> NO_STEPS = ImmutableList.of();

	@PrePlanProcessor
	public static List<DeploymentStep> validate(DeltaSpecification spec) {
		DeployedApplication deployedApplication = spec.getDeployedApplication();
		Set<String> conditions = getReleaseConditions(deployedApplication);

		if (conditions.isEmpty()) {
			return NO_STEPS;
		}

		VerificationResult result = validateReleaseConditions(conditions, deployedApplication.getVersion());
		if (result.failed()) {
			throw new IllegalArgumentException(buildErrorMessage(deployedApplication, result.getViolatedConditions()));
		}

		Builder<DeploymentStep> deploymentSteps = ImmutableList.builder();
		Environment environment = deployedApplication.getEnvironment();
		if (!environment.hasProperty(ENV_RECHECK_CONDITIONS_PROPERTY) || TRUE.equals(environment.getProperty(ENV_RECHECK_CONDITIONS_PROPERTY))) {
			int recheckConditionsAtDeploymentTimeOrder = environment.hasProperty(ENV_RECHECK_CONDITIONS_ORDER_PROPERTY) ? environment
			        .<Integer> getProperty(ENV_RECHECK_CONDITIONS_ORDER_PROPERTY) : DefaultOrders.PRE_FLIGHT;
			logger.debug("Adding release auth condition check step at order {}", recheckConditionsAtDeploymentTimeOrder);
			deploymentSteps.add(new LogReleaseConditionsStep(recheckConditionsAtDeploymentTimeOrder, conditions,
			        deployedApplication.getVersion()));
		}
		return deploymentSteps.build();
	}

	private static Set<String> getReleaseConditions(DeployedApplication deployedApplication) {
		final Environment environment = deployedApplication.getEnvironment();
		Collection<PropertyDescriptor> conditions = filter(environment.getType().getDescriptor().getPropertyDescriptors(), new Predicate<PropertyDescriptor>() {
			@Override
			public boolean apply(PropertyDescriptor input) {
				Object propertyValue = input.get(environment);
				// if the property is from release condition category and it is checked
				if(input.getCategory().equals(RELEASE_CHECKLIST_CATEGORY_NAME) && !input.getName().startsWith("requires")) {
					logger.warn(String.format("Ignoring property '%s' as a release condition for environment '%s' as it does not start with the keyword 'requires'", input.getName(), environment.getName()));
					return false;
				}
				return input.getCategory().equals(RELEASE_CHECKLIST_CATEGORY_NAME) && propertyValue != null && (Boolean) propertyValue;
			}
		});

		Collection<String> conditionNames = transform(conditions, new Function<PropertyDescriptor, String>() {
			@Override
			public String apply(PropertyDescriptor input) {
				return input.getName().replaceFirst("requires", "satisfies");
			}
		});
		return new HashSet<String>(conditionNames);
	}

	private static String buildErrorMessage(DeployedApplication deployedApplication, Set<ViolatedCondition<?>> violatedConditions) {
		StringBuilder errorMessage = new StringBuilder();
		errorMessage.append("Cannot deploy '").append(deployedApplication.getName()).append("' (version ")
		        .append(deployedApplication.getVersion().getVersion()).append(") to '").append(deployedApplication.getEnvironment().getName())
		        .append("' as the following release conditions are not met:");
		for (ViolatedCondition<?> violatedCondition : violatedConditions) {
			errorMessage.append("\n- '").append(violatedCondition.name).append("'");
		}
		return errorMessage.toString();
	}

    private static final Logger logger = LoggerFactory.getLogger(CheckReleaseConditionsAreMet.class);
}
