package com.xebialabs.deployit.plugins.releaseauth.upgrade;

import java.util.*;

import javax.jcr.*;
import javax.jcr.lock.LockException;
import javax.jcr.nodetype.ConstraintViolationException;
import javax.jcr.nodetype.NoSuchNodeTypeException;
import javax.jcr.nodetype.NodeType;
import javax.jcr.version.VersionException;

import org.slf4j.Logger;
import org.slf4j.LoggerFactory;

import com.google.common.base.Splitter;
import com.google.common.collect.Multimap;
import com.google.common.collect.Ordering;
import com.google.common.collect.TreeMultimap;

import com.xebialabs.deployit.plugin.api.reflect.DescriptorRegistry;
import com.xebialabs.deployit.plugin.api.reflect.Type;
import com.xebialabs.deployit.plugin.api.udm.Application;
import com.xebialabs.deployit.server.api.repository.RawRepository;
import com.xebialabs.deployit.server.api.upgrade.Upgrade;
import com.xebialabs.deployit.server.api.upgrade.UpgradeException;
import com.xebialabs.deployit.server.api.upgrade.Version;

import static com.google.common.collect.Lists.newArrayList;
import static com.google.common.collect.Sets.newTreeSet;

/**
 * This upgrader converts the deployment pipeline added to an application directly (release dashboard 3.7.x) to
 * a separate DeploymentPipeline CI under the Configuration node (release dashboard 3.8).
 */
public class Deployit38DeploymentPipeline extends Upgrade {
    /**
     * JCR constants from the 3.8 repository structure, see com.xebialabs.deployit.jcr.JcrConstants
     */
    public static final String DEPLOYIT_NAMESPACE_PREFIX = "deployit";
    public static final String CONFIGURATION_ITEM_NODETYPE_NAME = DEPLOYIT_NAMESPACE_PREFIX + ":configurationItem";
    public static final String METADATA_PROPERTY_NAME_PREFIX = "$";
    public static final String CONFIGURATION_ITEM_TYPE_PROPERTY_NAME = METADATA_PROPERTY_NAME_PREFIX + "configuration.item.type";

    private static final String CORE_DIRECTORY = "core.Directory";
    private static final String DEPLOYIT_37_DEPLOYMENT_PIPELINE_PROPERTY = "deploymentPipeline";

    @Override
    public boolean doUpgrade(RawRepository repository) throws UpgradeException {
        logger.info("*** Running Deployit 3.8 upgrade -- Externalize deployment pipeline ***");
        try {
            List<Type> applicationTypes = findAllApplicationSubtypes();
            List<Node> nodes = gatherNodes(repository, applicationTypes);
            Multimap<String, Node> directoryToApplications = divideApplicationsByDirectory(nodes);
            createDirectories(repository, directoryToApplications.keySet());
            for (Node node : directoryToApplications.values()) {
                convertPipelineForApplication(repository, node);
            }
            logger.info("*** Finished upgrade ***");
            return true;
        } catch (Exception e) {
            throw new UpgradeException("", e);
        }
    }

    private List<Type> findAllApplicationSubtypes() {
        List<Type> applicationTypes = newArrayList(Type.valueOf(Application.class));
        applicationTypes.addAll(DescriptorRegistry.getSubtypes(Type.valueOf(Application.class)));
        logger.info("All application subtypes: " + applicationTypes);
        return applicationTypes;
    }

    private List<Node> gatherNodes(final RawRepository repository, final List<Type> subtypes) {
        List<Node> nodes = newArrayList();

        for (Type subtype : subtypes) {
            nodes.addAll(repository.findNodesByType(subtype));
        }
        return nodes;
    }

    private Multimap<String, Node> divideApplicationsByDirectory(final List<Node> nodes) throws RepositoryException {
        Multimap<String, Node> result = TreeMultimap.create(Ordering.natural(), new Comparator<Node>() {
            @Override
            public int compare(final Node o1, final Node o2) {
                try {
                    return o1.getPath().compareTo(o2.getPath());
                } catch (RepositoryException e) {
                    throw new RuntimeException(e);
                }
            }
        });

        for (Node node : nodes) {
            result.put(node.getParent().getPath(), node);
        }

        return result;
    }

    private static void convertPipelineForApplication(RawRepository repository, Node applicationNode)
            throws Exception {
        logger.debug("Found node [{}] that is a subtype of Application", applicationNode.getPath());
        if (applicationNode.hasProperty(DEPLOYIT_37_DEPLOYMENT_PIPELINE_PROPERTY)) {
            logger.debug("Node [{}] has a deploymentPipeline property", applicationNode.getPath());

            Node pipelineNode = createPipelineNode(repository, applicationNode);
            updateApplication(applicationNode, pipelineNode);
        } else {
            logger.debug("Node [{}] does not have a deploymentPipeline property", applicationNode.getPath());
        }
    }

    private static void updateApplication(Node applicationNode, Node pipelineNode) throws Exception {
        applicationNode.setProperty(DEPLOYIT_37_DEPLOYMENT_PIPELINE_PROPERTY, (String[]) null);
        applicationNode.setProperty("pipeline", pipelineNode);
    }

    private static Node createPipelineNode(RawRepository repository, Node applicationNode) throws PathNotFoundException, RepositoryException {
        Property property = applicationNode.getProperty(DEPLOYIT_37_DEPLOYMENT_PIPELINE_PROPERTY);
        Value[] environments = property.getValues();

        Node pipelineNode = createNode(repository,
                applicationIdToConfigurationId(applicationNode) + "-pipeline", Type.valueOf("release.DeploymentPipeline"));
        pipelineNode.setProperty("pipeline", environments);
        return pipelineNode;
    }

    private static void createDirectories(RawRepository repository, Set<String> nodes) throws AccessDeniedException, ItemNotFoundException, RepositoryException {
        TreeSet<String> s = gatherAllIds(nodes);
        for (String node : s) {
            Node n = repository.read(node);
            if (isDirectory(n)) {
                cloneNodeUnderConfigurationRoot(repository, n);
            }
        }
    }

    protected static TreeSet<String> gatherAllIds(final Set<String> nodes) {
        TreeSet<String> toBeCreated = newTreeSet();
        for (String node : nodes) {
            Iterable<String> split = Splitter.on("/").omitEmptyStrings().split(node);
            StringBuilder n = new StringBuilder();
            for (String s : split) {
                n.append("/").append(s);
                toBeCreated.add(n.toString());
            }
        }
        return toBeCreated;
    }

    private static boolean exists(RawRepository repository, String id) {
        try {
            repository.read(id);
            return true;
        } catch(Exception e) {
            return false;
        }
    }

    private static void cloneNodeUnderConfigurationRoot(RawRepository repository, Node applicationDirectory) throws RepositoryException {
        String newId = applicationIdToConfigurationId(applicationDirectory);
        logger.debug("Cloning node [{}] under root [{}] at id [{}]", new Object[] { applicationDirectory.getPath(), "Configuration", newId });

        if (!exists(repository, newId)) {
            Node node = repository.create(newId);
            cloneMixins(applicationDirectory, node);
            cloneProperties(applicationDirectory, node);
        }
    }

    private static String applicationIdToConfigurationId(Node directory)
            throws RepositoryException {
        return "/Configuration" + directory.getPath().substring("/Applications".length());
    }

    private static void cloneProperties(Node src, Node dst) throws RepositoryException {
        for (@SuppressWarnings("rawtypes") Iterator iterator = src.getProperties(); iterator.hasNext();) {
            Property property = (Property) iterator.next();

            if (property.getDefinition().isProtected()) {
                continue;
            }

            if (property.isMultiple()) {
                dst.setProperty(property.getName(), property.getValues(), property.getType());
            } else {
                dst.setProperty(property.getName(), property.getValue(), property.getType());
            }
        }
    }

    private static void cloneMixins(Node src, Node dst) throws RepositoryException {
        for (NodeType mixin: src.getMixinNodeTypes()) {
            dst.addMixin(mixin.getName());
        }
    }

    private static boolean isDirectory(Node node) throws ValueFormatException,
            RepositoryException, PathNotFoundException {
        return node.getProperty(CONFIGURATION_ITEM_TYPE_PROPERTY_NAME).getString().equals(CORE_DIRECTORY);
    }

    private static Node createNode(RawRepository repository, String id, Type type) throws NoSuchNodeTypeException, VersionException, ConstraintViolationException, LockException, RepositoryException {
        Node node = repository.create(id);
        if (node.getIndex() != 1) {
            node.remove();
            throw new IllegalStateException("The impossible has happened, you already have a node called [" + id + "] in your repository. Please contact deployit-support@xebialabs.com");
        }

        node.addMixin(CONFIGURATION_ITEM_NODETYPE_NAME);
        node.addMixin(NodeType.MIX_REFERENCEABLE);
        node.addMixin(NodeType.MIX_VERSIONABLE);
        node.setProperty(CONFIGURATION_ITEM_TYPE_PROPERTY_NAME, type.toString());
        logger.debug("Created a [{}] at [{}]", CORE_DIRECTORY, id);
        return node;
    }

    @Override
    public Version upgradeVersion() {
        return Version.valueOf("deployment-checklist-plugin", "3.8.0");
    }

    private static final Logger logger = LoggerFactory.getLogger(Deployit38DeploymentPipeline.class);
}
