package com.xebialabs.deployit.plugin.generic.container;

import com.google.common.collect.Maps;
import com.google.common.collect.Sets;
import com.xebialabs.deployit.plugin.api.deployment.planning.Contributor;
import com.xebialabs.deployit.plugin.api.deployment.planning.DeploymentPlanningContext;
import com.xebialabs.deployit.plugin.api.deployment.specification.Delta;
import com.xebialabs.deployit.plugin.api.deployment.specification.Deltas;
import com.xebialabs.deployit.plugin.api.deployment.specification.Operation;
import com.xebialabs.deployit.plugin.api.udm.Deployed;
import com.xebialabs.deployit.plugin.generic.ci.Container;
import com.xebialabs.deployit.plugin.generic.deployed.AbstractDeployed;
import com.xebialabs.deployit.plugin.generic.step.ScriptExecutionStep;
import com.xebialabs.deployit.plugin.generic.step.WaitStep;

import java.util.List;
import java.util.Map;
import java.util.Set;

import static com.google.common.base.Preconditions.checkArgument;
import static com.google.common.base.Strings.nullToEmpty;

public class LifeCycleContributor {

    @Contributor
    public void restartContainers(Deltas deltas, DeploymentPlanningContext result) {
        Set<Container> targets = gatherTargets(deltas.getDeltas());
        for (Container target : targets) {
            if (!nullToEmpty(target.getRestartScript()).trim().isEmpty()) {
                addStep(target.getRestartScript(), target.getRestartOrder(), target, "Restarting", result);
                addWaitStep(target.getRestartWaitTime(), target.getRestartOrder(), target, "restart", result);
            } else {
                checkArgument(!nullToEmpty(target.getStartScript()).trim().isEmpty(), "start script must be specified when no restart script defined.");
                checkArgument(!nullToEmpty(target.getStopScript()).trim().isEmpty(), "stop script must be specified when no restart script defined.");
                addStep(target.getStartScript(), target.getStartOrder(), target, "Starting", result);
                addWaitStep(target.getStartWaitTime(), target.getStartOrder(), target, "start", result);
                addStep(target.getStopScript(), target.getStopOrder(), target, "Stopping", result);
                addWaitStep(target.getStopWaitTime(), target.getStopOrder(), target, "stop", result);
            }
        }
    }

	private void addWaitStep(int waitTime, int scriptOrder, Container target, String action, DeploymentPlanningContext result) {
		if (waitTime > 0) {
			result.addStep(new WaitStep(scriptOrder + 1, waitTime, target.getName(), action));
		}
	}

    private void addStep(String script, int scriptOrder, Container container, String verb, DeploymentPlanningContext result) {
        Map<String, Object> vars = Maps.newHashMap();
        vars.put("container", container);
        result.addStep(new ScriptExecutionStep(scriptOrder,script, container, vars, getDescription(verb, container)));
    }

    private String getDescription(String verb, Container container) {
        return String.format("%s %s", verb, container.getName());
    }


    private Set<Container> gatherTargets(List<Delta> operations) {
        final Set<Container> targets = Sets.newTreeSet();
        for (Delta operation : operations) {
            addTarget(targets, operation.getOperation(),  operation.getDeployed());
            addTarget(targets, operation.getOperation(), operation.getPrevious());
        }
        return targets;
    }

    private void addTarget(Set<Container> targets, Operation operation, Deployed<?, ?> deployed) {
        if (deployed == null)
            return;
        if (deployed instanceof AbstractDeployed) {
            AbstractDeployed<?> abstractDeployed = (AbstractDeployed<?>) deployed;
            boolean restartRequired = abstractDeployed.isRestartRequired();
            if (restartRequired) {
                if (operation != Operation.NOOP) {
                    targets.add((Container) deployed.getContainer());
                } else if (abstractDeployed.isRestartRequiredForNoop()) {
                    targets.add((Container) deployed.getContainer());
                }
            }
        }
    }
}
