package com.xebialabs.deployit.plugin.generic.ci;

import java.util.Collections;
import java.util.List;
import java.util.Map;
import java.util.Set;

import com.xebialabs.deployit.plugin.api.flow.Step;
import com.xebialabs.deployit.plugin.api.inspection.Inspect;
import com.xebialabs.deployit.plugin.api.inspection.InspectionContext;
import com.xebialabs.deployit.plugin.api.udm.Metadata;
import com.xebialabs.deployit.plugin.api.udm.Property;
import com.xebialabs.deployit.plugin.api.udm.base.BaseContainer;
import com.xebialabs.deployit.plugin.generic.step.InspectScriptExecutionStep;
import com.xebialabs.deployit.plugin.overthere.Host;

import static com.google.common.base.Strings.emptyToNull;
import static com.google.common.collect.Lists.newArrayList;
import static com.google.common.collect.Maps.newHashMap;
import static com.google.common.collect.Sets.newHashSet;
import static com.xebialabs.deployit.plugin.generic.freemarker.ConfigurationHolder.resolveExpression;

@SuppressWarnings("serial")
@Metadata(virtual = true, description = "A container to which generic CIs can be deployed. Start, stop and restart behavior of this container can be controlled using the corresponding script properties.")
public class Container extends BaseContainer implements GenericContainer {

    protected Map<String, Object> freeMarkerContext = Collections.singletonMap("container", (Object) this);

    @Property(asContainment = true, description = "Host upon which the container resides")
    private Host host;

    @Property(required = false, hidden = true, description = "Classpath to the script used to start the generic container.")
    private String startScript;

    @Property(required = false, hidden = true, description = "Classpath to the script used to stop the generic container.")
    private String stopScript;

    @Property(required = false, hidden = true, description = "Classpath to the script used to restart the generic container.")
    private String restartScript;

    @Property(defaultValue = "90", hidden = true, description = "The order of the start container step in the step list.")
    private int startOrder;

    @Property(defaultValue = "10", hidden = true, description = "The order of the stop container step in the step list.")
    private int stopOrder;

    @Property(defaultValue = "90", hidden = true, description = "The order of the restart container step in the step list.")
    private int restartOrder;

    @Property(required = false, defaultValue = "0", hidden = true, description = "The time to wait in seconds for a container restart action.")
    private int restartWaitTime;

    @Property(defaultValue = "0", hidden = true, description = "The time to wait in seconds for a container start action.")
    private int startWaitTime;

    @Property(defaultValue = "0", hidden = true, description = "The time to wait in seconds for a container stop action.")
    private int stopWaitTime;

    @Property(required = false, description = "Environment variables for container", label="Environment Variables")
    private Map<String, String> envVars = newHashMap();

    @Property(required = false, hidden = true, description = "Classpath to the script used to inspect the generic container.")
    private String inspectScript;

    @Property(hidden = true, required = false, description = "Additional classpath resources that should be uploaded to the working directory before executing the inspect script.")
    private Set<String> inspectClasspathResources = newHashSet();

    @Property(hidden = true, required = false, description = "Additional template classpath resources that should be uploaded to the working directory before executing the inspect script." +
            "The template is first rendered and the rendered content copied to a file, with the same name as the template, in the working directory.")
    private Set<String> inspectTemplateClasspathResources = newHashSet();

    public List<Step> controlTaskDispatch(String name, Map<String,String> args) {
        return ControlTaskDelegate.dispatch(name, args, this, this);
    }

    public Map<String, String> getEnvVars() {
         return envVars;
    }

    public void setEnvVars(Map<String, String> envVars) {
        this.envVars = envVars;
    }

    @Inspect
    public void inspectContainer(InspectionContext ctx) {
        if (emptyToNull(getInspectScript()) != null) {
            InspectScriptExecutionStep step = new InspectScriptExecutionStep(this, getInspectScript(), getHost(), freeMarkerContext, "Inspect " + this);
            step.setTemplateClasspathResources(newArrayList(getInspectTemplateClasspathResources()));
            step.setClasspathResources(newArrayList(getInspectClasspathResources()));
            ctx.addStep(step);
        }
    }

    @Override
    public Host getHost() {
        return host;
    }

    @Override
    public <T extends GenericContainer> T getParentContainer() {
        // The top Level Container does not have a parent
        return null;
    }


    public void setHost(Host host) {
        this.host = host;
    }

    public String getStartScript() {
        return resolveExpression(startScript, freeMarkerContext);
    }

    public void setStartScript(String startScript) {
        this.startScript = startScript;
    }

    public String getStopScript() {
        return resolveExpression(stopScript, freeMarkerContext);
    }

    public void setStopScript(String stopScript) {
        this.stopScript = stopScript;
    }

    public String getRestartScript() {
        return resolveExpression(restartScript, freeMarkerContext);
    }

    public void setRestartScript(String restartScript) {
        this.restartScript = restartScript;
    }

    public int getStartOrder() {
        return startOrder;
    }

    public void setStartOrder(int startOrder) {
        this.startOrder = startOrder;
    }

    public int getStopOrder() {
        return stopOrder;
    }

    public void setStopOrder(int stopOrder) {
        this.stopOrder = stopOrder;
    }

    public int getRestartOrder() {
        return restartOrder;
    }

    public void setRestartOrder(int restartOrder) {
        this.restartOrder = restartOrder;
    }

    public int getRestartWaitTime() {
        return restartWaitTime;
    }

    public void setRestartWaitTime(int restartWaitTime) {
        this.restartWaitTime = restartWaitTime;
    }

    public int getStartWaitTime() {
        return startWaitTime;
    }

    public void setStartWaitTime(int startWaitTime) {
        this.startWaitTime = startWaitTime;
    }

    public int getStopWaitTime() {
        return stopWaitTime;
    }

    public void setStopWaitTime(int stopWaitTime) {
        this.stopWaitTime = stopWaitTime;
    }

    public String getInspectScript() {
        return resolveExpression(inspectScript, freeMarkerContext);
    }

    public void setInspectScript(String inspectScript) {
        this.inspectScript = inspectScript;
    }

    public Set<String> getInspectClasspathResources() {
        return resolveExpression(inspectClasspathResources, freeMarkerContext);
    }

    public void setInspectClasspathResources(Set<String> inspectClasspathResources) {
        this.inspectClasspathResources = inspectClasspathResources;
    }

    public Set<String> getInspectTemplateClasspathResources() {
        return resolveExpression(inspectTemplateClasspathResources, freeMarkerContext);
    }

    public void setInspectTemplateClasspathResources(Set<String> inspectTemplateClasspathResources) {
        this.inspectTemplateClasspathResources = inspectTemplateClasspathResources;
    }
}
