package com.xebialabs.deployit.plugin.generic.step;

import java.util.Set;

import com.google.common.base.Strings;

import com.xebialabs.deployit.plugin.api.flow.StepExitCode;
import com.xebialabs.deployit.plugin.api.udm.DeployableArtifact;
import com.xebialabs.deployit.plugin.overthere.HostContainer;
import com.xebialabs.overthere.OverthereFile;

import static com.google.common.base.Preconditions.checkNotNull;
import static com.google.common.base.Strings.nullToEmpty;
import static com.google.common.collect.Sets.newHashSet;

@SuppressWarnings("serial")
public class ArtifactDeleteStep extends BaseDeploymentStep {

    private DeployableArtifact artifact;
    private String targetDirectory;
    private boolean targetDirectoryShared;
    private String targetFile;
    private String description;
    private Set<String> deleteAdditionalTargetFiles = newHashSet();

    public ArtifactDeleteStep(int order, HostContainer container, DeployableArtifact artifact, String targetDirectory) {
        super(order, container);
        this.artifact = artifact;
        this.targetDirectory = checkNotNull(targetDirectory);
    }

    public ArtifactDeleteStep(int order, HostContainer container, String targetDirectory) {
        this(order, container, null, targetDirectory);
    }

    @Override
    protected StepExitCode doExecute() throws Exception {
        if (artifact != null) {
            deleteArtifact();
        } else {
            deleteTargetFile();
        }

        deleteAdditionalFiles(getDeleteAdditionalTargetFiles());

        return StepExitCode.SUCCESS;
    }

    protected void deleteTargetFile() {
        OverthereFile remoteFile = getRemoteConnection().getFile(targetDirectory);
        if (!nullToEmpty(targetFile).trim().isEmpty()) {
            remoteFile = remoteFile.getFile(targetFile);
        }
        deleteFile(remoteFile);
    }

    protected void deleteAdditionalFiles(Set<String> files) {
        for (String file : files) {
            OverthereFile remoteFile = getRemoteConnection().getFile(file);
            if (remoteFile.exists()) {
                getCtx().logOutput("Deleting " + remoteFile.getPath() + " on host " + getContainer().getHost());
                remoteFile.deleteRecursively();
            } else {
                getCtx().logOutput(remoteFile.getPath() + " does not exist on host " + getContainer().getHost() + ". Will not perform delete.");
            }
        }
    }

    protected void deleteArtifact() {
        OverthereFile localFile = artifact.getFile();
        OverthereFile remoteDir = getRemoteConnection().getFile(targetDirectory);

        if (localFile.isDirectory()) {
            if (targetDirectoryShared) {
                for (OverthereFile file : localFile.listFiles()) {
                    deleteFile(remoteDir.getFile(file.getName()));
                }
            } else {
                deleteFile(remoteDir);
            }
        } else {
            String name = nullToEmpty(targetFile).trim().isEmpty() ? artifact.getName() : targetFile;
            deleteFile(remoteDir.getFile(name));
        }
    }

    protected void deleteFile(OverthereFile file) {
        getCtx().logOutput("Deleting " + file.getPath() + " on host " + getContainer().getHost());
        if (file.exists()) {
            file.deleteRecursively();
        } else {
            getCtx().logOutput("File " + file.getPath() + " was already gone from " + getContainer().getHost());
        }

    }

    @Override
    public String getDescription() {
        if (description == null) {
            return generateDescription();
        }
        return description;
    }

    protected String generateDescription() {
        if (artifact != null) {
            return "Delete " + artifact.getName() + " from " + getContainer().getHost();
        }

        if (!Strings.nullToEmpty(targetFile).trim().isEmpty()) {
            return "Delete file " + targetFile + " from directory " + targetDirectory + " on host " +  getContainer().getHost();
        }

        return "Delete directory " + targetDirectory + " from  " + getContainer().getHost();
    }

    public void setDescription(String description) {
        this.description = description;
    }

    public String getTargetFile() {
        return targetFile;
    }

    public void setTargetFile(String targetFile) {
        this.targetFile = targetFile;
    }

    public boolean isTargetDirectoryShared() {
        return targetDirectoryShared;
    }

    public void setTargetDirectoryShared(boolean targetDirectoryShared) {
        this.targetDirectoryShared = targetDirectoryShared;
    }

    public Set<String> getDeleteAdditionalTargetFiles() {
        return deleteAdditionalTargetFiles;
    }

    public void setDeleteAdditionalTargetFiles(Set<String> deleteAdditionalTargetFiles) {
        this.deleteAdditionalTargetFiles = deleteAdditionalTargetFiles;
    }
}
