package com.xebialabs.deployit.plugin.generic.step;

import java.io.IOException;
import java.io.StringWriter;
import java.util.List;
import java.util.Map;
import javax.mail.MessagingException;
import org.slf4j.MDC;
import com.google.common.base.Joiner;
import com.google.common.base.Preconditions;
import com.google.common.base.Strings;

import com.xebialabs.deployit.plugin.api.flow.StepExitCode;
import com.xebialabs.deployit.plugin.generic.freemarker.ConfigurationHolder;
import com.xebialabs.deployit.plugin.mail.SmtpServer;
import com.xebialabs.deployit.plugin.overthere.HostContainer;

import freemarker.template.Configuration;
import freemarker.template.Template;
import freemarker.template.TemplateException;

import static com.google.common.collect.Lists.newArrayList;
import static com.google.common.collect.Maps.newHashMap;

@SuppressWarnings("serial")
public class InstructionStep extends BaseDeploymentStep {

    private static final String MDC_KEY_TEMPLATE_PATH = "templatePath";

    private Map<String, Object> vars;

    private String templatePath;

    private boolean paused = false;

    private List<String> toAddresses = newArrayList();

    private String subject;

    private String fromAddress;

    private SmtpServer mailServer;

    public InstructionStep(int order, String description, HostContainer container, Map<String, Object> vars, String templatePath) {
        super(order, description, container);
        this.vars = newHashMap(vars);
        this.vars.put("step",this);
        this.templatePath = templatePath;
        Preconditions.checkNotNull(templatePath);
    }

    @Override
    protected StepExitCode doExecute() throws Exception {
        MDC.put(MDC_KEY_TEMPLATE_PATH, templatePath);
        try {
            String instructions = renderEmailTemplate(templatePath, vars);
            if (paused) {
                getCtx().logOutput("Assuming manual process performed. Continuing...");
                getCtx().logOutput("------------");
                getCtx().logOutput("Instructions");
                getCtx().logOutput("------------");
                getCtx().logOutput(instructions);
                return StepExitCode.SUCCESS;
            }

            mailInstructions(instructions);
            getCtx().logOutput(instructions);
            paused = true;
            return StepExitCode.PAUSE;
        } finally {
            MDC.remove(MDC_KEY_TEMPLATE_PATH);
        }
    }

    protected void mailInstructions(String instructions) {
        if (!toAddresses.isEmpty() && mailServer != null) {
            getCtx().logOutput("Mailing instructions to " + Joiner.on(',').join(toAddresses));
            subject = (Strings.isNullOrEmpty(subject)) ? getDescription() : subject;
            try {
                mailServer.sendMessage(subject, instructions, toAddresses, fromAddress);
            } catch (MessagingException e) {
                getCtx().logError("Failed to send mail.",e);
                getCtx().logOutput(Strings.repeat("-",50));
            }
        }
    }

    protected String renderEmailTemplate(String template, Map<String, Object> vars) throws IOException, TemplateException {
        Configuration cfg = ConfigurationHolder.getConfiguration();
        Template loadedTemplate = cfg.getTemplate(template);
        StringWriter sw = new StringWriter();
        loadedTemplate.process(vars, sw);
        return sw.toString();
    }

    public List<String> getToAddresses() {
        return toAddresses;
    }

    public void setToAddresses(List<String> toAddresses) {
        this.toAddresses = toAddresses;
    }

    public String getSubject() {
        return subject;
    }

    public void setSubject(String subject) {
        this.subject = subject;
    }

    public String getFromAddress() {
        return fromAddress;
    }

    public void setFromAddress(String fromAddress) {
        this.fromAddress = fromAddress;
    }

    public SmtpServer getMailServer() {
        return mailServer;
    }

    public void setMailServer(SmtpServer mailServer) {
        this.mailServer = mailServer;
    }
}

