#
# Copyright (c) 2018. All rights reserved.
#
# This software and all trademarks, trade names, and logos included herein are the property of XebiaLabs, Inc. and its affiliates, subsidiaries, and licensors.
#

function Get-WebIsapiFilters($websitePath)
{
    Get-WebConfiguration -PSPath $websitePath -Filter "/system.webServer/isapiFilters/filter"
}

$websites = Get-Website
if($websites) {
    foreach ($website in $websites) {
        $id = "$($container.id)/$($website.name)"
        discovered $id $prototype.type
    
        inspectedProperty $id "physicalPath" $website.physicalPath
        inspectedProperty $id "applicationPoolName" $website.applicationPool
        inspectedProperty $id "websiteId" $website.id

        inspectedProperty $id "limitsConnectionTimeout" $website.limits.connectionTimeout.TotalSeconds

        inspectedProperty $id "logFileEnabled" ([string]$website.logFile.enabled).ToLower()
        inspectedProperty $id "logFileLogFormat" $website.logFile.logFormat
        inspectedProperty $id "logFileLogExtFileFlags" $website.logFile.logExtFileFlags
        inspectedProperty $id "logFileDirectory" $website.logFile.directory
        inspectedProperty $id "logFilePeriod" $website.logFile.period
        if ($website.logFile.period -eq "MaxSize") {
            inspectedProperty $id "logFileTruncateSize" $website.logFile.truncateSize
        }
        inspectedProperty $id "logFileLocalTimeRollover" ([string]$website.logFile.localTimeRollover).ToLower()

        inspectDirectoryBrowsing $id "IIS:\Sites\$($website.name)"

        inspectedItem $id

        inspectAuthentication $id "IIS:\Sites\$($website.name)"

        foreach ($binding in (Get-WebBinding -Name $website.name)) {
            $bid = "$id/$($binding.protocol)_$($binding.bindingInformation -replace "[:*]", "_")"
            discovered $bid "iis.WebsiteBinding"

            $parts = $binding.bindingInformation.split(":")
            Assert {$parts.length -eq 3} "Web binding expected to be in format 'ipAddress:port:hostHeader' but was [$($binding.bindingInformation)]"
    
            inspectedProperty $bid "protocol" $binding.protocol
            inspectedProperty $bid "ipAddress" $parts[0]
            inspectedProperty $bid "port" $parts[1]
            inspectedProperty $bid "hostHeader" $parts[2]
            
            inspectedItem $bid
        }
        $websitePath = "IIS:\Sites\$($website.name)"
        # ISAPI filters
        $currentFilters = Get-WebIsapiFilters $websitePath
        $parentFilters = Get-WebIsapiFilters "iis:\sites"

        if ($parentFilters -ne $Null) {
            Compare-Object $currentFilters $parentFilters | ForEach-Object {
                $filter = $_.InputObject
                $fid = "$id/$($filter.name)"

                discovered $fid "iis.ISAPIFilter"
                inspectedProperty $fid "filterName" $filter.name
                inspectedProperty $fid "executable" $filter.path

                inspectedItem $fid
            }
        } ElseIf ($currentFilters -ne $Null) {
            Foreach ($filter in $currentFilters) {
                $fid = "$id/$($filter.name)"

                discovered $fid "iis.ISAPIFilter"
                inspectedProperty $fid "filterName" $filter.name
                inspectedProperty $fid "executable" $filter.path

                inspectedItem $fid
            }
        }
    }
}
