#
# Copyright (c) 2018. All rights reserved.
#
# This software and all trademarks, trade names, and logos included herein are the property of XebiaLabs, Inc. and its affiliates, subsidiaries, and licensors.
#

Function convertServiceAccountToIdentityType($serviceAccount) {
    $serviceAccount = $serviceAccount.toLower()
    $serviceAccountType = 2   #networkservice
    if ($serviceAccount -eq "localsystem") {
        $serviceAccountType = 0
    } elseif ($serviceAccount -eq "localservice") {
        $serviceAccountType = 1
    }
    return $serviceAccountType
}

# Get provided application pool name or use the deployed name.
$applicationPoolName = if($deployed.applicationPoolName) { $deployed.applicationPoolName } else { $deployed.name }

# Verify application pool name is not being changed.
if($previousDeployed) {
    $previousApplicationPoolName = if($previousDeployed.applicationPoolName) { $previousDeployed.applicationPoolName } else { $previousDeployed.name }
    if($applicationPoolName -ne $previousApplicationPoolName) {
        Write-Host "Renaming an application pool is not supported. Undeploy and deploy the application pool instead."
        Exit 1
    }
}

# Verify username and password are provided if serviceAccount if set to 'SpecificUser'.
if ($deployed.serviceAccount -eq "SpecificUser") {
    if (!$deployed.username) {
        throw "Username is required when service account is set to [SpecificUser]."
    }
    if (!$deployed.password) {
        throw "Password is required when service account is set to [SpecificUser]."
    }
}

# Enable 32bit mode on all application pools
Write-Host "WARN: Setting Enable32BitAppOnWin64 for IIS 6 means that all worker processes (all application pools) are running as 32 bit processes. All deployed pools must have the same value of this property."
$appPools = New-Object System.DirectoryServices.DirectoryEntry("IIS://localhost/W3SVC/AppPools")
$appPools.Enable32BitAppOnWin64 = $deployed.enable32BitAppOnWin64


# Check whether application pool already exists and then either retrieve or create it.
$appPoolSetting = Get-WmiObject -Namespace "ROOT\MicrosoftIISv2" -Class IIsApplicationPoolSetting -Filter "Name ='W3SVC/APPPOOLS/$applicationPoolName'"
if ($appPoolSetting) {
    Write-Host "Modifying existing application pool [$applicationPoolName]."
} else {
    Write-Host "Creating new application pool [$applicationPoolName]."

    $appPoolSetting = ([WMICLASS]"ROOT\MicrosoftIISv2:IIsApplicationPoolSetting").CreateInstance()
    $appPoolSetting.Name = "W3SVC/AppPools/$applicationPoolName"
}

# Set application pool properties.
if ($deployed.serviceAccount -eq "SpecificUser") {
    $appPoolSetting.AppPoolIdentityType = 3
    $appPoolSetting.WAMUsername = $deployed.username
    $appPoolSetting.WAMUserPass = [System.Text.Encoding]::UTF8.GetString([System.Convert]::FromBase64String($deployed.password))
} elseif ($deployed.serviceAccount) {
    $appPoolSetting.AppPoolIdentityType = convertServiceAccountToIdentityType $deployed.serviceAccount
}

# Save application pool.
$appPoolSetting.Put() | Out-Null
$appPools.CommitChanges()

# Stop application pool so that we can configure it correctly before starting it again from start-application-pool.ps1
$appPool = Get-WmiObject -Namespace "ROOT\MicrosoftIISv2" -Class IIsApplicationPool -Filter "Name ='W3SVC/APPPOOLS/$applicationPoolName'"
$appPool.Stop()

