package com.xebialabs.deployit.plugin.jbossas.step;

import static com.google.common.collect.Maps.newHashMap;

import java.io.IOException;
import java.io.StringWriter;
import java.util.Collections;
import java.util.Map;

import org.apache.commons.lang.StringUtils;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;

import com.xebialabs.deployit.plugin.api.execution.Step;
import com.xebialabs.deployit.plugin.api.inspection.InspectionExecutionContext;
import com.xebialabs.deployit.plugin.generic.ci.Container;
import com.xebialabs.deployit.plugin.generic.freemarker.CiAwareObjectWrapper;
import com.xebialabs.deployit.plugin.generic.freemarker.ConfigurationHolder;
import com.xebialabs.deployit.plugin.generic.freemarker.FileUploader;
import com.xebialabs.deployit.plugin.overthere.ExecutionContextOverthereProcessOutputHandler;
import com.xebialabs.overthere.CmdLine;
import com.xebialabs.overthere.OperatingSystemFamily;
import com.xebialabs.overthere.OverthereConnection;
import com.xebialabs.overthere.OverthereFile;
import com.xebialabs.overthere.RuntimeIOException;
import com.xebialabs.overthere.util.MultipleOverthereProcessOutputHandler;
import com.xebialabs.overthere.util.OverthereUtils;

import freemarker.template.Configuration;
import freemarker.template.TemplateException;

public class TwiddleStep {

	private transient OverthereConnection remoteConn;
	private transient OverthereFile remoteWorkingDir;

	private final Container container;
	private final String scriptTemplatePath;
	protected InspectionExecutionContext ctx;

	public TwiddleStep(Container container, String scriptTemplatePath) {
		this.container = container;
		this.scriptTemplatePath = scriptTemplatePath;
	}

	public Step.Result execute(InspectionExecutionContext ctx) throws Exception {
		this.ctx = ctx;
		String osSpecificTemplate = resolveOsSpecificTemplate();
		String executableContent = evaluateTemplate(osSpecificTemplate, getVars());
		logger.debug(executableContent);
		OverthereFile executable = uploadExecutable(executableContent, StringUtils.substringAfterLast(osSpecificTemplate, "/"));
		CmdLine cmdLine = CmdLine.build(executable.getPath());
		TwiddleProcessOutputHandler parser = new TwiddleProcessOutputHandler();
		MultipleOverthereProcessOutputHandler handle = MultipleOverthereProcessOutputHandler.multiHandler(
		        new ExecutionContextOverthereProcessOutputHandler(ctx), parser);
		ctx.logOutput("Executing " + executable.getPath() + " on host " + container.getHost());
		int rc = getRemoteConnection().execute(handle, cmdLine);
		if (rc == 0) {
			doHandle(parser);
			return Step.Result.Success;
		}
		return Step.Result.Fail;
	}

	protected Map<String, Object> getVars() {
		return Collections.emptyMap();
	}

	protected void doHandle(TwiddleProcessOutputHandler handler) {
	}

	private String resolveOsSpecificTemplate() {
		String scriptExt = StringUtils.substringAfterLast(scriptTemplatePath, ".");
		String osSpecificScript = scriptTemplatePath;

		if (StringUtils.isBlank(scriptExt)) {
			OperatingSystemFamily os = container.getHost().getOs();
			osSpecificScript = osSpecificScript + os.getScriptExtension();
		}
		if (!classpathResourceExists(osSpecificScript)) {
			throw new IllegalArgumentException("Resource " + osSpecificScript + " not found in classpath");
		}

		return osSpecificScript;
	}

	private String evaluateTemplate(String templatePath, Map<String, Object> vars) {
		Configuration cfg = ConfigurationHolder.getConfiguration();
		try {
			freemarker.template.Template template = cfg.getTemplate(templatePath);
			StringWriter sw = new StringWriter();
			template.createProcessingEnvironment(vars, sw, new CiAwareObjectWrapper(new WorkingFolderUploader())).process();
			return sw.toString();
		} catch (IOException e) {
			throw new RuntimeIOException(e);
		} catch (TemplateException e) {
			logger.error("evaluate Template error", e);
			throw new RuntimeException(e);
		}
	}

	private OverthereConnection getRemoteConnection() {
		if (remoteConn == null) {
			remoteConn = container.getHost().getConnection();
		}
		return remoteConn;
	}

	private OverthereFile getRemoteWorkingDirectory() {
		if (remoteWorkingDir == null) {
			OverthereFile tempDir = getRemoteConnection().getTempFile("jboss_plugin", ".tmp");
			tempDir.mkdir();
			remoteWorkingDir = tempDir;
		}
		return remoteWorkingDir;
	}

	private OverthereFile uploadExecutable(String content, String fileName) {
		OverthereFile targetExecutable = getRemoteWorkingDirectory().getFile(fileName);
		OverthereUtils.write(content.getBytes(), targetExecutable);
		targetExecutable.setExecutable(true);
		return targetExecutable;
	}

	private boolean classpathResourceExists(String resource) {
		return Thread.currentThread().getContextClassLoader().getResource(resource) != null;
	}

	private class WorkingFolderUploader implements FileUploader {
		private Map<String, String> uploadedFiles = newHashMap();

		public String upload(OverthereFile file) {
			if (uploadedFiles.containsKey(file.getName())) {
				return uploadedFiles.get(file.getName());
			}
			uploadedFiles.put(file.getName(), file.getPath());
			return file.getPath();
		}
	}

	public Container getContainer() {
		return container;
	}

	protected final Logger logger = LoggerFactory.getLogger(this.getClass());

}
